/* Unit test suite for functions SystemParametersInfo and GetSystemMetrics.

   Copyright 2002 Andriy Palamarchuk

*/

#include <windows.h>
#include <stdlib.h>
#include <stdio.h>
#include "wt_helper.h"

#define SPI_SETBEEP_REGKEY           "Control Panel\\Sound"
#define SPI_SETBEEP_VALNAME          "Beep"
#define SPI_SETMOUSE_REGKEY             "Control Panel\\Mouse"
#define SPI_SETMOUSE_VALNAME1           "MouseThreshold1"
#define SPI_SETMOUSE_VALNAME2           "MouseThreshold2"
#define SPI_SETMOUSE_VALNAME3           "MouseSpeed"
#define SPI_SETBORDER_REGKEY         "Control Panel\\Desktop"
#define SPI_SETBORDER_VALNAME        "BorderWidth"
#define SPI_SETKEYBOARDSPEED_REGKEY             "Control Panel\\Keyboard"
#define SPI_SETKEYBOARDSPEED_VALNAME            "KeyboardSpeed"
#define SPI_ICONHORIZONTALSPACING_REGKEY        "Control Panel\\Desktop"
#define SPI_ICONHORIZONTALSPACING_VALNAME       "IconSpacing"
#define SPI_SETSCREENSAVETIMEOUT_REGKEY         "Control Panel\\Desktop"
#define SPI_SETSCREENSAVETIMEOUT_VALNAME        "ScreenSaveTimeOut"
#define SPI_SETSCREENSAVEACTIVE_REGKEY          "Control Panel\\Desktop"
#define SPI_SETSCREENSAVEACTIVE_VALNAME         "ScreenSaveActive"
#define SPI_SETGRIDGRANULARITY_REGKEY           "Control Panel\\Desktop"
#define SPI_SETGRIDGRANULARITY_VALNAME          "GridGranularity"
#define SPI_SETKEYBOARDDELAY_REGKEY             "Control Panel\\Keyboard"
#define SPI_SETKEYBOARDDELAY_VALNAME            "KeyboardDelay"
#define SPI_ICONVERTICALSPACING_REGKEY          "Control Panel\\Desktop"
#define SPI_ICONVERTICALSPACING_VALNAME         "IconVerticalSpacing"
#define SPI_SETICONTITLEWRAP_REGKEY             "Control Panel\\Desktop"
#define SPI_SETICONTITLEWRAP_VALNAME            "IconTitleWrap"
#define SPI_SETMENUDROPALIGNMENT_REGKEY         "Software\\Microsoft\\Windows NT\\CurrentVersion\\Windows"
#define SPI_SETMENUDROPALIGNMENT_VALNAME        "MenuDropAlignment"
#define SPI_SETDOUBLECLKWIDTH_REGKEY            "Control Panel\\Mouse"
#define SPI_SETDOUBLECLKWIDTH_VALNAME           "DoubleClickWidth"
#define SPI_SETDOUBLECLKHEIGHT_REGKEY           "Control Panel\\Mouse"
#define SPI_SETDOUBLECLKHEIGHT_VALNAME          "DoubleClickHeight"
#define SPI_SETDOUBLECLICKTIME_REGKEY           "Control Panel\\Mouse"
#define SPI_SETDOUBLECLICKTIME_VALNAME          "DoubleClickSpeed"
#define SPI_SETMOUSEBUTTONSWAP_REGKEY           "Control Panel\\Mouse"
#define SPI_SETMOUSEBUTTONSWAP_VALNAME          "SwapMouseButtons"
#define SPI_SETWORKAREA_REGKEY                  "Control Panel\\Desktop"
#define SPI_SETWORKAREA_VALNAME                 "WINE_WorkArea"
#define SPI_SETSHOWSOUNDS_REGKEY     "Control Panel\\Accessibility\\ShowSounds"
#define SPI_SETSHOWSOUNDS_VALNAME    "On"
#define SPI_SETFASTTASKSWITCH_REGKEY            "Control Panel\\Desktop"
#define SPI_SETFASTTASKSWITCH_VALNAME           "CoolSwitch"
#define SPI_SETDRAGFULLWINDOWS_REGKEY           "Control Panel\\Desktop"
#define SPI_SETDRAGFULLWINDOWS_VALNAME          "DragFullWindows"
/* FIXME - don't have access to Windows with this action (W95, NT5.0). Set real values */
#define SPI_SETKEYBOARDPREF_REGKEY      "Control Panel\\Desktop"
#define SPI_SETKEYBOARDPREF_VALNAME     "WINE_WorkArea"
#define SPI_SETSCREENREADER_REGKEY      "Control Panel\\Desktop"
#define SPI_SETSCREENREADER_VALNAME     "???"

/* volatile registry branch under CURRENT_USER_REGKEY for temporary values storage */
#define WINE_CURRENT_USER_REGKEY     "Wine"

HWND ghTestWnd;

DWORD WINAPI SysParamsThreadFunc( LPVOID lpParam );
LRESULT CALLBACK SysParamsTestWndProc( HWND hWnd, UINT msg, WPARAM wParam,
                                       LPARAM lParam );
int win_major_version;
int win_minor_version;
int win_platform_id;

int change_counter;
int change_last_param;
char change_reg_section[MAX_PATH];

LRESULT CALLBACK SysParamsTestWndProc( HWND hWnd, UINT msg, WPARAM wParam,
                                       LPARAM lParam )
{
    switch(msg) {

    case WM_SETTINGCHANGE:
	change_counter++;
	change_last_param = wParam;
        strncpy( change_reg_section, (LPSTR) lParam, sizeof( change_reg_section ));
        break;

    case WM_DESTROY:
        PostQuitMessage( 0 );
        break;

    default:
        return( DefWindowProc( hWnd, msg, wParam, lParam ) );
    }

    return 0;
}

/*
Performs testing for system parameters messages
params:
 - system parameter id
 - supposed value of the registry key
*/
void test_change_message( UINT action, char *reg_section )  {
    ok( 1 == change_counter, "Each message is processed" );
    change_counter = 0;
    ok( action == change_last_param, "Correct action is processed" );
    change_last_param = 0;
    ok( !strcmp( reg_section, change_reg_section ),
        "Unexpected registry section" );
    strcpy( change_reg_section, "");
}

/*
 * Tests the HKEY_CURRENT_USER subkey value.
 * The value should contain string value.
 *
 * Params:
 * lpsSubKey - subkey name
 * lpsRegName - registry entry name
 * lpsTestValue - value to test
 */
void _test_reg_key( LPSTR subKey, LPSTR valName, LPSTR testValue,
                    char *file, int line ) {
    CHAR  value[MAX_PATH] = "";
    DWORD valueLen = MAX_PATH;
    DWORD type;
    HKEY hKey;

    RegOpenKey( HKEY_CURRENT_USER, subKey, &hKey );
    RegQueryValueEx( hKey, valName, NULL, &type, value, &valueLen );
    RegCloseKey( hKey );
    if(strcmp( testValue, value ))
        printf( "subKey: %s, valName: %s, testValue: %s, value: %s "
                "called from %s, line %i\n",
                subKey, valName, testValue, value, file, line);
    ok( !strcmp( testValue, value ),
        "System parameter registry key expected value" );
}

#define test_reg_key( subKey, valName, testValue ) \
    _test_reg_key( subKey, valName, testValue, __FILE__, __LINE__ )

void test_SPI_SETBEEP( void )                   /*      2 */
{
    BOOL old_b;
    BOOL b;
    BOOL curr_val;
    
    SystemParametersInfo( SPI_GETBEEP, 0, &old_b, 0 );

    curr_val = TRUE;
    SystemParametersInfo( SPI_SETBEEP, curr_val, 0, SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETBEEP, "" );
    test_reg_key( SPI_SETBEEP_REGKEY,
                  SPI_SETBEEP_VALNAME,
                  curr_val ? "Yes" : "No" );
    SystemParametersInfo( SPI_GETBEEP, 0, &b, 0 );
    ok( curr_val == b, "retrieved value is the same as set" );
    if (has_unicode())
    {
        SystemParametersInfoW( SPI_GETBEEP, 0, &b, 0 );
        ok( curr_val == b, "retrieved value is the same as set" );
    }
    
    /* is a message sent for the second change? */
    SystemParametersInfo( SPI_SETBEEP, curr_val, 0, SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETBEEP, "" );

    curr_val = FALSE;
    if (has_unicode())
        SystemParametersInfoW( SPI_SETBEEP, curr_val, 0, SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    else
        SystemParametersInfo( SPI_SETBEEP, curr_val, 0, SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETBEEP, "" );
    test_reg_key( SPI_SETBEEP_REGKEY,
                  SPI_SETBEEP_VALNAME,
                  curr_val ? "Yes" : "No" );
    SystemParametersInfo( SPI_GETBEEP, 0, &b, 0 );
    ok( curr_val == b, "retrieved value is the same as set" );
    if (has_unicode())
    {
        SystemParametersInfoW( SPI_GETBEEP, 0, &b, 0 );
        ok( curr_val == b, "retrieved value is the same as set" );
    }
    ok( MessageBeep( MB_OK ), "Return value of MessageBeep when sound is disabled" );

    SystemParametersInfo( SPI_SETBEEP, old_b, 0, SPIF_UPDATEINIFILE );
}

char *setmouse_valuenames[3] = {
    SPI_SETMOUSE_VALNAME1,
    SPI_SETMOUSE_VALNAME2,
    SPI_SETMOUSE_VALNAME3
};

/*
 * Runs check for one setting of spi_setmouse.
 */
void run_spi_setmouse_test( int curr_val[], POINT *req_change, POINT *proj_change,
                            int nchange )
{
    INT mi[3];
    static int aw_turn = 0;

    char buf[20];
    int i;
    POINT mv;

    aw_turn++;
    if (has_unicode() && (aw_turn % 2))        /* call unicode version each second call */
        SystemParametersInfoW( SPI_SETMOUSE, 0, curr_val, SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    else
        SystemParametersInfo( SPI_SETMOUSE, 0, curr_val, SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETMOUSE, "" );
    for (i = 0; i < 3; i++)
    {
        sprintf( buf, "%d", curr_val[i] );
        test_reg_key( SPI_SETMOUSE_REGKEY, setmouse_valuenames[i], buf );
    }

    SystemParametersInfo( SPI_GETMOUSE, 0, mi, 0 );
    for (i = 0; i < 3; i++)
    {
        ok(mi[i] == curr_val[i], "retrieved value is the same as set");
    }

    if (has_unicode())
    {
        SystemParametersInfoW( SPI_GETMOUSE, 0, mi, 0 );
        for (i = 0; i < 3; i++)
        {
            ok(mi[i] == curr_val[i], "retrieved value is the same as set");
        }
    }

    for (i = 0; i < nchange; i++)
    {
        mouse_event( MOUSEEVENTF_ABSOLUTE | MOUSEEVENTF_MOVE, 0, 0, 0, 0 );
        mouse_event( MOUSEEVENTF_MOVE, req_change[i].x, req_change[i].y, 0, 0 );
        GetCursorPos( &mv );
        ok( proj_change[i].x == mv.x, "Projectected dx and real dx comparison. "
            "May fail under high load." );
        ok( proj_change[i].y == mv.y, "Projectected dy equals real dy" 
            "May fail under high load." );
    }
}

void test_SPI_SETMOUSE( void )                  /*      4 */
{
    INT old_mi[3];

    /* win nt default values - 6, 10, 1*/
    INT curr_val[3] = {6, 10, 1};

    /*requested and projected mouse movements */
    POINT req_change[] =   { {6, 6}, { 7, 6}, { 8, 6}, {10, 10}, {11, 10}, {100, 100} };
    POINT proj_change1[] = { {6, 6}, {14, 6}, {16, 6}, {20, 20}, {22, 20}, {200, 200} };
    POINT proj_change2[] = { {6, 6}, {14, 6}, {16, 6}, {20, 20}, {44, 20}, {400, 400} };
    POINT proj_change3[] = { {6, 6}, {14, 6}, {16, 6}, {20, 20}, {22, 20}, {200, 200} };
    POINT proj_change4[] = { {6, 6}, { 7, 6}, { 8, 6}, {10, 10}, {11, 10}, {100, 100} };
    POINT proj_change5[] = { {6, 6}, { 7, 6}, {16, 6}, {20, 20}, {22, 20}, {200, 200} };
    POINT proj_change6[] = { {6, 6}, {28, 6}, {32, 6}, {40, 40}, {44, 40}, {400, 400} };
    POINT proj_change7[] = { {6, 6}, {14, 6}, {32, 6}, {40, 40}, {44, 40}, {400, 400} };
    POINT proj_change8[] = { {6, 6}, {28, 6}, {32, 6}, {40, 40}, {44, 40}, {400, 400} };

    int nchange = sizeof( req_change ) / sizeof( POINT );
    
    SystemParametersInfo( SPI_GETMOUSE, 0, old_mi, 0 );

    run_spi_setmouse_test( curr_val, req_change, proj_change1, nchange );

    /* acceleration change */
    curr_val[2] = 2;
    run_spi_setmouse_test( curr_val, req_change, proj_change2, nchange );

    /* acceleration change */
    curr_val[2] = 3;
    run_spi_setmouse_test( curr_val, req_change, proj_change3, nchange );

    /* acceleration change */
    curr_val[2] = 0;
    run_spi_setmouse_test( curr_val, req_change, proj_change4, nchange );

    /* threshold change */
    curr_val[2] = 1;
    curr_val[0] = 7;
    run_spi_setmouse_test( curr_val, req_change, proj_change5, nchange );

    /* threshold change */
    curr_val[2] = 2;
    curr_val[0] = 6;
    curr_val[1] = 6;
    run_spi_setmouse_test( curr_val, req_change, proj_change6, nchange );

    /* threshold change */
    curr_val[1] = 7;
    run_spi_setmouse_test( curr_val, req_change, proj_change7, nchange );

    /* threshold change */
    curr_val[1] = 5;
    run_spi_setmouse_test( curr_val, req_change, proj_change8, nchange );

    SystemParametersInfo( SPI_SETMOUSE, 0, old_mi, SPIF_UPDATEINIFILE );
}

void test_SPI_SETBORDER( void )                 /*      6 */
{
    UINT old_border;
    UINT curr_val;
    UINT border;
    UINT frame;
    char buf[10];

    /* tests one configuration of border settings */
#define TEST_SETBORDER \
    SystemParametersInfo( SPI_SETBORDER, curr_val, 0, SPIF_UPDATEINIFILE | SPIF_SENDCHANGE ); \
    test_change_message( SPI_SETBORDER, "" ); \
    sprintf( buf, "%d", curr_val ); \
    test_reg_key( SPI_SETBORDER_REGKEY, SPI_SETBORDER_VALNAME, buf ); \
    if (curr_val == 0) curr_val = 1; \
    SystemParametersInfo( SPI_GETBORDER, 0, &border, 0 ); \
    ok( border == border, "retrieved value is the same as set" ); \
    frame = curr_val + GetSystemMetrics( SM_CXDLGFRAME ); \
    ok( frame == GetSystemMetrics( SM_CXFRAME ), "In synch with SM_CXFRAME" ); \
    ok( frame == GetSystemMetrics( SM_CYFRAME ), "In synch with SM_CYFRAME" ); \
    ok( frame == GetSystemMetrics( SM_CXSIZEFRAME ), "In synch with SM_CXSIZEFRAME" ); \
    ok( frame == GetSystemMetrics( SM_CYSIZEFRAME ), "In synch with SM_CYSIZEFRAME" )

    /* These tests hang when XFree86 4.0 for Windows is running (tested on WinNT, SP2,
       Cygwin/XFree 4.1.0. Skip the test when XFree86 is running. */
    if ((get_w32_impl() == W32_MS) && !FindWindow( NULL, "Cygwin/XFree86" ))
    {
        SystemParametersInfo( SPI_GETBORDER, 0, &old_border, 0 );

        curr_val = 1;
        TEST_SETBORDER;

        curr_val = 0;
        TEST_SETBORDER;

        curr_val = 7;
        TEST_SETBORDER;

        curr_val = 20;
        TEST_SETBORDER;
    
        /* This will restore sane values if the test hang previous run. */
        if ( old_border == 7 || old_border == 20 )
            old_border = 1;

        SystemParametersInfo( SPI_SETBORDER, old_border, 0, SPIF_UPDATEINIFILE );
    }
    
#undef TEST_SETBORDER
}

void test_SPI_SETKEYBOARDSPEED( void )          /*     10 */
{
    UINT old_speed;
    UINT curr_val;
    UINT speed;
    char buf[10];

    SystemParametersInfo( SPI_GETKEYBOARDSPEED, 0, &old_speed, 0 );

    curr_val = 0;
    SystemParametersInfo( SPI_SETKEYBOARDSPEED, curr_val, 0,
                           SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETKEYBOARDSPEED, "" );
    sprintf( buf, "%d", curr_val );
    test_reg_key( SPI_SETKEYBOARDSPEED_REGKEY, SPI_SETKEYBOARDSPEED_VALNAME, buf );

    SystemParametersInfo( SPI_GETKEYBOARDSPEED, 0, &speed, 0 );
    ok( curr_val == speed, "retrieved value is the same as set" );

    curr_val = 32;
    SystemParametersInfo( SPI_SETKEYBOARDSPEED, curr_val, 0,
                          SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    curr_val = 31; /* max value */
    test_change_message( SPI_SETKEYBOARDSPEED, "" );
    sprintf( buf, "%d", curr_val );
    test_reg_key( SPI_SETKEYBOARDSPEED_REGKEY, SPI_SETKEYBOARDSPEED_VALNAME, buf );

    SystemParametersInfo( SPI_GETKEYBOARDSPEED, 0, &speed, 0 );
    ok( curr_val == speed, "retrieved value is the same as set" );

    SystemParametersInfo( SPI_SETKEYBOARDSPEED, old_speed, 0, SPIF_UPDATEINIFILE );
}

void test_SPI_ICONHORIZONTALSPACING( void )     /*     13 */
{
    UINT old_spacing;
    UINT spacing;
    UINT curr_val;
    char buf[10];

    /* default value: 75 */
    SystemParametersInfo( SPI_ICONHORIZONTALSPACING, 0, &old_spacing, 0 );

    curr_val = 101;
    SystemParametersInfo( SPI_ICONHORIZONTALSPACING, curr_val, 0,
                          SPIF_UPDATEINIFILE | SPIF_SENDCHANGE);
    test_change_message( SPI_ICONHORIZONTALSPACING, "" );
    sprintf( buf, "%d", curr_val );
    test_reg_key( SPI_ICONHORIZONTALSPACING_REGKEY,
                  SPI_ICONHORIZONTALSPACING_VALNAME, buf );

    SystemParametersInfo( SPI_ICONHORIZONTALSPACING, 0, &spacing, 0 );
    ok( curr_val == spacing, "retrieved value is the same as set" );
    ok( curr_val == GetSystemMetrics( SM_CXICONSPACING ),
        "in synch with SM_CXICONSPACING" );
    
    curr_val = 10;
    SystemParametersInfo( SPI_ICONHORIZONTALSPACING, curr_val, 0,
                          SPIF_UPDATEINIFILE | SPIF_SENDCHANGE);
    curr_val = 32;      /*min value*/
    test_change_message( SPI_ICONHORIZONTALSPACING, "" );
    sprintf( buf, "%d", curr_val );
    test_reg_key( SPI_ICONHORIZONTALSPACING_REGKEY,
                  SPI_ICONHORIZONTALSPACING_VALNAME, buf );

    SystemParametersInfo( SPI_ICONHORIZONTALSPACING, 0, &spacing, 0 );
    ok( curr_val == spacing, "retrieved value is the same as set" );
    ok( curr_val == GetSystemMetrics( SM_CXICONSPACING ),
        "in synch with SM_CXICONSPACING" );

    SystemParametersInfo( SPI_ICONHORIZONTALSPACING, old_spacing, 0,
                          SPIF_UPDATEINIFILE );
}

void test_SPI_SETSCREENSAVETIMEOUT( void )      /*     14 */
{
    UINT old_timeout;
    UINT timeout;
    UINT curr_val;
    char buf[10];

    SystemParametersInfo( SPI_GETSCREENSAVETIMEOUT, 0, &old_timeout, 0 );

    curr_val = 0;
    SystemParametersInfo( SPI_SETSCREENSAVETIMEOUT, curr_val, 0,
                           SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETSCREENSAVETIMEOUT, "" );
    sprintf( buf, "%d", curr_val );
    test_reg_key( SPI_SETSCREENSAVETIMEOUT_REGKEY,
                  SPI_SETSCREENSAVETIMEOUT_VALNAME, buf );

    SystemParametersInfo( SPI_GETSCREENSAVETIMEOUT, 0, &timeout, 0 );
    ok( curr_val == timeout, "retrieved value is the same as set" );

    curr_val = 50000;
    SystemParametersInfo( SPI_SETSCREENSAVETIMEOUT, curr_val, 0,
                          SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETSCREENSAVETIMEOUT, "" );
    sprintf( buf, "%d", curr_val );
    test_reg_key( SPI_SETSCREENSAVETIMEOUT_REGKEY,
                  SPI_SETSCREENSAVETIMEOUT_VALNAME, buf );

    SystemParametersInfo( SPI_GETSCREENSAVETIMEOUT, 0, &timeout, 0 );
    ok( curr_val == timeout, "retrieved value is the same as set" );

    SystemParametersInfo( SPI_SETSCREENSAVETIMEOUT, old_timeout, 0,
                          SPIF_UPDATEINIFILE );
}

void test_SPI_SETSCREENSAVEACTIVE( void )       /*     17 */
{
    BOOL old_b;
    BOOL b;
    BOOL curr_val;

    SystemParametersInfo( SPI_GETSCREENSAVEACTIVE, 0, &old_b, 0 );

    curr_val = TRUE;
    SystemParametersInfo( SPI_SETSCREENSAVEACTIVE, curr_val, 0,
                           SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETSCREENSAVEACTIVE, "" );
    test_reg_key( SPI_SETSCREENSAVEACTIVE_REGKEY,
                  SPI_SETSCREENSAVEACTIVE_VALNAME,
                  curr_val ? "1" : "0" );

    SystemParametersInfo( SPI_GETSCREENSAVEACTIVE, 0, &b, 0 );
    ok( curr_val == b, "retrieved value is the same as set" );

    curr_val = FALSE;
    SystemParametersInfo( SPI_SETSCREENSAVEACTIVE, curr_val, 0,
                          SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETSCREENSAVEACTIVE, "" );
    test_reg_key( SPI_SETSCREENSAVEACTIVE_REGKEY,
                  SPI_SETSCREENSAVEACTIVE_VALNAME,
                  curr_val ? "1" : "0" );

    SystemParametersInfo( SPI_GETSCREENSAVEACTIVE, 0, &b, 0 );
    ok( curr_val == b, "retrieved value is the same as set" );

    SystemParametersInfo( SPI_SETSCREENSAVEACTIVE, old_b, 0, SPIF_UPDATEINIFILE );
}

void test_SPI_SETGRIDGRANULARITY( void )        /*     19 */
{
    /* ??? */;
}

void test_SPI_SETKEYBOARDDELAY( void )          /*     23 */
{
    UINT old_speed;
    UINT speed;
    UINT curr_val;
    char buf[10];

    SystemParametersInfo( SPI_GETKEYBOARDSPEED, 0, &old_speed, 0 );

    curr_val = 0;
    SystemParametersInfo( SPI_SETKEYBOARDSPEED, curr_val, 0,
                           SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETKEYBOARDSPEED, "" );
    sprintf( buf, "%d", curr_val );
    test_reg_key( SPI_SETKEYBOARDSPEED_REGKEY,
                  SPI_SETKEYBOARDSPEED_VALNAME, buf );

    SystemParametersInfo( SPI_GETKEYBOARDSPEED, 0, &speed, 0 );
    ok( curr_val == speed, "retrieved value is the same as set" );

    curr_val = 32;
    SystemParametersInfo( SPI_SETKEYBOARDSPEED, curr_val, 0,
                          SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    curr_val = 31;      /* max value */
    test_change_message( SPI_SETKEYBOARDSPEED, "" );
    sprintf( buf, "%d", curr_val );
    test_reg_key( SPI_SETKEYBOARDSPEED_REGKEY,
                  SPI_SETKEYBOARDSPEED_VALNAME, buf );

    SystemParametersInfo( SPI_GETKEYBOARDSPEED, 0, &speed, 0 );
    ok( curr_val == speed, "retrieved value is the same as set" );

    SystemParametersInfo( SPI_SETKEYBOARDSPEED, old_speed, 0, SPIF_UPDATEINIFILE );
}

void test_SPI_ICONVERTICALSPACING( void )       /*     24 */
{
    UINT old_spacing;
    UINT spacing;
    UINT curr_val;
    char buf[10];

    /* default value: 75 */
    SystemParametersInfo( SPI_ICONVERTICALSPACING, 0, &old_spacing, 0 );

    curr_val = 101;
    SystemParametersInfo( SPI_ICONVERTICALSPACING, curr_val, 0,
                          SPIF_UPDATEINIFILE | SPIF_SENDCHANGE);
    test_change_message( SPI_ICONVERTICALSPACING, "" );
    sprintf( buf, "%d", curr_val );
    test_reg_key( SPI_ICONVERTICALSPACING_REGKEY,
                  SPI_ICONVERTICALSPACING_VALNAME, buf );

    SystemParametersInfo( SPI_ICONVERTICALSPACING, 0, &spacing, 0 );
    ok( curr_val == spacing, "retrieved value is the same as set" );
    ok( curr_val == GetSystemMetrics( SM_CYICONSPACING ),
        "in synch with SM_CYICONSPACING" );
    
    curr_val = 10;
    SystemParametersInfo( SPI_ICONVERTICALSPACING, curr_val, 0,
                          SPIF_UPDATEINIFILE | SPIF_SENDCHANGE);
    curr_val = 32;      /*min value*/
    test_change_message( SPI_ICONVERTICALSPACING, "" );
    sprintf( buf, "%d", curr_val );
    test_reg_key( SPI_ICONVERTICALSPACING_REGKEY,
                  SPI_ICONVERTICALSPACING_VALNAME, buf );

    SystemParametersInfo( SPI_ICONVERTICALSPACING, 0, &spacing, 0 );
    ok( curr_val == spacing, "retrieved value is the same as set" );
    ok( curr_val == GetSystemMetrics( SM_CYICONSPACING ),
        "in synch with SM_CYICONSPACING" );

    SystemParametersInfo( SPI_ICONVERTICALSPACING, old_spacing, 0,
                          SPIF_UPDATEINIFILE );
}

void test_SPI_SETICONTITLEWRAP( void )          /*     26 */
{
    BOOL old_b;
    BOOL b;
    BOOL curr_val;

    /* These tests hang when XFree86 4.0 for Windows is running (tested on WinNT, SP2,
       Cygwin/XFree 4.1.0. Skip the test when XFree86 is running. */
    if ((get_w32_impl() == W32_MS) && !FindWindow( NULL, "Cygwin/XFree86" ))
    {
        SystemParametersInfo( SPI_GETICONTITLEWRAP, 0, &old_b, 0 );

        curr_val = TRUE;
        SystemParametersInfo( SPI_SETICONTITLEWRAP, curr_val, 0,
                              SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
        test_change_message( SPI_SETICONTITLEWRAP, "" );
        test_reg_key( SPI_SETICONTITLEWRAP_REGKEY,
                      SPI_SETICONTITLEWRAP_VALNAME,
                      curr_val ? "1" : "0" );

        SystemParametersInfo( SPI_GETICONTITLEWRAP, 0, &b, 0 );
        ok( curr_val == b, "retrieved value is the same as set" );

        curr_val = FALSE;
        SystemParametersInfo( SPI_SETICONTITLEWRAP, curr_val, 0,
                              SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
        test_change_message( SPI_SETICONTITLEWRAP, "" );
        test_reg_key( SPI_SETICONTITLEWRAP_REGKEY,
                      SPI_SETICONTITLEWRAP_VALNAME,
                      curr_val ? "1" : "0" );

        SystemParametersInfo( SPI_GETICONTITLEWRAP, 0, &b, 0 );
        ok( curr_val == b, "retrieved value is the same as set" );

        SystemParametersInfo( SPI_SETICONTITLEWRAP, old_b, 0, SPIF_UPDATEINIFILE );
    }
}

void test_SPI_SETMENUDROPALIGNMENT( void )      /*     28 */
{
    BOOL old_b;
    BOOL b;
    BOOL curr_val;

    SystemParametersInfo( SPI_GETMENUDROPALIGNMENT, 0, &old_b, 0 );

    curr_val = TRUE;
    SystemParametersInfo( SPI_SETMENUDROPALIGNMENT, curr_val, 0,
                           SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETMENUDROPALIGNMENT, "" );
    test_reg_key( SPI_SETMENUDROPALIGNMENT_REGKEY,
                  SPI_SETMENUDROPALIGNMENT_VALNAME,
                  curr_val ? "1" : "0" );

    SystemParametersInfo( SPI_GETMENUDROPALIGNMENT, 0, &b, 0 );
    ok( curr_val == b, "retrieved value is the same as set" );
    ok( curr_val == GetSystemMetrics( SM_MENUDROPALIGNMENT ),
        "In synch with SM_MENUDROPALIGNMENT" );

    curr_val = FALSE;
    SystemParametersInfo( SPI_SETMENUDROPALIGNMENT, curr_val, 0,
                          SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETMENUDROPALIGNMENT, "" );
    test_reg_key( SPI_SETMENUDROPALIGNMENT_REGKEY,
                  SPI_SETMENUDROPALIGNMENT_VALNAME,
                  curr_val ? "1" : "0" );

    SystemParametersInfo( SPI_GETMENUDROPALIGNMENT, 0, &b, 0 );
    ok( curr_val == b, "retrieved value is the same as set" );
    ok( curr_val == GetSystemMetrics( SM_MENUDROPALIGNMENT ),
        "In synch with SM_MENUDROPALIGNMENT" );

    SystemParametersInfo( SPI_SETMENUDROPALIGNMENT, old_b, 0,
                           SPIF_UPDATEINIFILE );
}

void test_SPI_SETDOUBLECLKWIDTH( void )         /*     29 */
{
    UINT old_width;
    UINT curr_val;
    char buf[10];

    old_width = GetSystemMetrics( SM_CXDOUBLECLK );

    curr_val = 0;
    SystemParametersInfo( SPI_SETDOUBLECLKWIDTH, curr_val, 0,
                           SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETDOUBLECLKWIDTH, "" );
    sprintf( buf, "%d", curr_val );
    test_reg_key( SPI_SETDOUBLECLKWIDTH_REGKEY,
                  SPI_SETDOUBLECLKWIDTH_VALNAME, buf );
    ok( curr_val == GetSystemMetrics( SM_CXDOUBLECLK ),
        "retrieved value is the same as set" );

    curr_val = 10000;
    SystemParametersInfo( SPI_SETDOUBLECLKWIDTH, curr_val, 0,
                           SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETDOUBLECLKWIDTH, "" );
    sprintf( buf, "%d", curr_val );
    test_reg_key( SPI_SETDOUBLECLKWIDTH_REGKEY,
                  SPI_SETDOUBLECLKWIDTH_VALNAME, buf );
    ok( curr_val == GetSystemMetrics( SM_CXDOUBLECLK ),
        "retrieved value is the same as set" );

    SystemParametersInfo( SPI_SETDOUBLECLKWIDTH, old_width, 0,
                          SPIF_UPDATEINIFILE );
}

void test_SPI_SETDOUBLECLKHEIGHT( void )        /*     30 */
{
    UINT old_height;
    UINT curr_val;
    char buf[10];

    old_height = GetSystemMetrics( SM_CYDOUBLECLK );

    curr_val = 0;
    SystemParametersInfo( SPI_SETDOUBLECLKHEIGHT, curr_val, 0,
                           SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETDOUBLECLKHEIGHT, "" );
    sprintf( buf, "%d", curr_val );
    test_reg_key( SPI_SETDOUBLECLKHEIGHT_REGKEY,
                  SPI_SETDOUBLECLKHEIGHT_VALNAME, buf );
    ok( curr_val == GetSystemMetrics( SM_CYDOUBLECLK ),
        "retrieved value is the same as set" );

    curr_val = 10000;
    SystemParametersInfo( SPI_SETDOUBLECLKHEIGHT, curr_val, 0,
                           SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETDOUBLECLKHEIGHT, "" );
    sprintf( buf, "%d", curr_val );
    test_reg_key( SPI_SETDOUBLECLKHEIGHT_REGKEY,
                  SPI_SETDOUBLECLKHEIGHT_VALNAME, buf );
    ok( curr_val == GetSystemMetrics( SM_CYDOUBLECLK ),
        "retrieved value is the same as set" );

    SystemParametersInfo( SPI_SETDOUBLECLKHEIGHT, old_height, 0,
                          SPIF_UPDATEINIFILE );
}

void test_SPI_SETDOUBLECLICKTIME( void )        /*     32 */
{
    UINT curr_val;
    UINT saved_val;
    UINT old_time;
    char buf[10];

    old_time = GetDoubleClickTime();

    curr_val = 0;
    SystemParametersInfo( SPI_SETDOUBLECLICKTIME, curr_val, 0,
                           SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETDOUBLECLICKTIME, "" );
    sprintf( buf, "%d", curr_val );
    test_reg_key( SPI_SETDOUBLECLICKTIME_REGKEY,
                  SPI_SETDOUBLECLICKTIME_VALNAME, buf );
    curr_val = 500; /* used value for 0 */
    ok( curr_val ==  GetDoubleClickTime(),
        "retrieved value is the same as set" );

    curr_val = 1000;
    SystemParametersInfo( SPI_SETDOUBLECLICKTIME, curr_val, 0,
                          SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETDOUBLECLICKTIME, "" );
    sprintf( buf, "%d", curr_val );
    test_reg_key( SPI_SETDOUBLECLICKTIME_REGKEY,
                  SPI_SETDOUBLECLICKTIME_VALNAME, buf );
    ok( curr_val ==  GetDoubleClickTime(),
        "retrieved value is the same as set" );

    saved_val = curr_val;

    curr_val = 0;
    SetDoubleClickTime( curr_val );
    sprintf( buf, "%d", saved_val );
    test_reg_key( SPI_SETDOUBLECLICKTIME_REGKEY,
                  SPI_SETDOUBLECLICKTIME_VALNAME, buf );
    curr_val = 500; /* used value for 0 */
    ok( curr_val ==  GetDoubleClickTime(),
        "retrieved value is the same as set" );

    curr_val = 1000;
    SetDoubleClickTime( curr_val );
    sprintf( buf, "%d", saved_val );
    test_reg_key( SPI_SETDOUBLECLICKTIME_REGKEY,
                  SPI_SETDOUBLECLICKTIME_VALNAME, buf );
    ok( curr_val ==  GetDoubleClickTime(),
        "retrieved value is the same as set" );

    SystemParametersInfo(SPI_SETDOUBLECLICKTIME, old_time, 0, SPIF_UPDATEINIFILE);
}

void test_SPI_SETMOUSEBUTTONSWAP( void )        /*     33 */
{
    BOOL old_b;
    BOOL curr_val;

    old_b = GetSystemMetrics( SM_SWAPBUTTON );

    curr_val = TRUE;
    SystemParametersInfo( SPI_SETMOUSEBUTTONSWAP, curr_val, 0,
                           SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETMOUSEBUTTONSWAP, "" );
    test_reg_key( SPI_SETMOUSEBUTTONSWAP_REGKEY,
                  SPI_SETMOUSEBUTTONSWAP_VALNAME,
                  curr_val ? "1" : "0" );
    ok( curr_val == GetSystemMetrics( SM_SWAPBUTTON ),
        "In synch with SM_SWAPBUTTON" );

    curr_val = FALSE;
    SystemParametersInfo( SPI_SETMOUSEBUTTONSWAP, curr_val, 0,
                          SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETMOUSEBUTTONSWAP, "" );
    test_reg_key( SPI_SETMOUSEBUTTONSWAP_REGKEY,
                  SPI_SETMOUSEBUTTONSWAP_VALNAME,
                  curr_val ? "1" : "0" );
    ok( curr_val == GetSystemMetrics( SM_SWAPBUTTON ),
        "In synch with SM_SWAPBUTTON" );

    SystemParametersInfo( SPI_SETMOUSEBUTTONSWAP, old_b, 0,
                           SPIF_UPDATEINIFILE );
}

void test_SPI_SETFASTTASKSWITCH( void )         /*     36 */
{
    /* Temporarily commented out. 
    BOOL old_b;
    BOOL b;
    BOOL res;
    BOOL curr_val;

    SystemParametersInfo( SPI_GETFASTTASKSWITCH, 0, &old_b, 0 );

    curr_val = TRUE;
    res = SystemParametersInfo( SPI_SETFASTTASKSWITCH, curr_val, 0,
                                 SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    ok( !res, "Must not set" );

    SystemParametersInfo( SPI_GETFASTTASKSWITCH, 0, &b, 0 );
    ok( b, "retrieved value is the same as set" );

    curr_val = FALSE;
    res = SystemParametersInfo( SPI_SETFASTTASKSWITCH, curr_val, 0,
                                 SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    ok( !res, "Must not set" );

    SystemParametersInfo( SPI_GETFASTTASKSWITCH, 0, &b, 0 );
        
    ok( b, "retrieved value is the same as set" );

    SystemParametersInfo( SPI_SETFASTTASKSWITCH, old_b, 0,
                           SPIF_UPDATEINIFILE );
    */
}

void test_SPI_SETDRAGFULLWINDOWS( void )        /*     37 */
{
    BOOL old_b;
    BOOL b;
    BOOL curr_val;

    SystemParametersInfo( SPI_GETDRAGFULLWINDOWS, 0, &old_b, 0 );

    curr_val = TRUE;
    SystemParametersInfo( SPI_SETDRAGFULLWINDOWS, curr_val, 0,
                           SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETDRAGFULLWINDOWS, "" );
    test_reg_key( SPI_SETDRAGFULLWINDOWS_REGKEY,
                  SPI_SETDRAGFULLWINDOWS_VALNAME,
                  curr_val ? "1" : "0" );

    SystemParametersInfo( SPI_GETDRAGFULLWINDOWS, 0, &b, 0 );
    ok( curr_val == b, "retrieved value is the same as set" );

    curr_val = FALSE;
    SystemParametersInfo( SPI_SETDRAGFULLWINDOWS, curr_val, 0,
                          SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETDRAGFULLWINDOWS, "" );
    test_reg_key( SPI_SETDRAGFULLWINDOWS_REGKEY,
                  SPI_SETDRAGFULLWINDOWS_VALNAME,
                  curr_val ? "1" : "0" );

    SystemParametersInfo( SPI_GETDRAGFULLWINDOWS, 0, &b, 0 );
    ok( curr_val == b, "retrieved value is the same as set" );

    SystemParametersInfo( SPI_SETDRAGFULLWINDOWS, old_b, 0, SPIF_UPDATEINIFILE );
}

void test_SPI_SETWORKAREA( void )               /*     47 */
{
    /* default - 0,0,1152, 864 */
    RECT old_area;
    RECT area;
    RECT curr_val;

    SystemParametersInfo(SPI_GETWORKAREA, 0, &old_area, 0);

    curr_val.left = 1;
    curr_val.top = 1;
    curr_val.right = 800;
    curr_val.bottom = 600;
    SystemParametersInfo( SPI_SETWORKAREA, 0, &curr_val,
                           SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETWORKAREA, "" );
    SystemParametersInfo( SPI_GETWORKAREA, 0, &area, 0 );
    ok( curr_val.left    == area.left,
        "retrieved value is the same as set" );
    ok( curr_val.top     == area.top,
       "retrieved value is the same as set" );
    ok( curr_val.right   == area.right,
        "retrieved value is the same as set" );
    ok( curr_val.bottom  == area.bottom,
        "retrieved value is the same as set" );

    curr_val.left = 2;
    curr_val.top = 2;
    curr_val.right = 801;
    curr_val.bottom = 601;
    SystemParametersInfo( SPI_SETWORKAREA, 0, &curr_val,
                          SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETWORKAREA, "" );
    SystemParametersInfo( SPI_GETWORKAREA, 0, &area, 0 );
    ok( curr_val.left    == area.left,
        "retrieved value is the same as set" );
    ok( curr_val.top     == area.top,
       "retrieved value is the same as set" );
    ok( curr_val.right   == area.right,
        "retrieved value is the same as set" );
    ok( curr_val.bottom  == area.bottom,
        "retrieved value is the same as set" );

    SystemParametersInfo(SPI_SETWORKAREA, 0, &old_area, SPIF_UPDATEINIFILE);
}

void test_SPI_SETSHOWSOUNDS( void )             /*     57 */
{
    BOOL old_b;
    BOOL b;
    BOOL curr_val;

    SystemParametersInfo( SPI_GETSHOWSOUNDS, 0, &old_b, 0 );

    curr_val = TRUE;
    SystemParametersInfo( SPI_SETSHOWSOUNDS, curr_val, 0,
                           SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETSHOWSOUNDS, "" );
    test_reg_key( SPI_SETSHOWSOUNDS_REGKEY,
                  SPI_SETSHOWSOUNDS_VALNAME,
                  curr_val ? "1" : "0" );

    SystemParametersInfo( SPI_GETSHOWSOUNDS, 0, &b, 0 );
    ok( curr_val == b, "retrieved value is the same as set" );
    ok( curr_val == GetSystemMetrics( SM_SHOWSOUNDS ),
        "in synch with SM_SHOWSOUNDS" );

    curr_val = FALSE;
    SystemParametersInfo( SPI_SETSHOWSOUNDS, curr_val, 0,
                          SPIF_UPDATEINIFILE | SPIF_SENDCHANGE );
    test_change_message( SPI_SETSHOWSOUNDS, "" );
    test_reg_key( SPI_SETSHOWSOUNDS_REGKEY,
                  SPI_SETSHOWSOUNDS_VALNAME,
                  curr_val ? "1" : "0" );

    SystemParametersInfo( SPI_GETSHOWSOUNDS, 0, &b, 0 );
    ok( curr_val == b, "retrieved value is the same as set" );
    ok( curr_val == GetSystemMetrics( SM_SHOWSOUNDS ),
        "in synch with SM_SHOWSOUNDS" );

    SystemParametersInfo( SPI_SETSHOWSOUNDS, old_b, 0, SPIF_UPDATEINIFILE );
}

void test_SPI_SETKEYBOARDPREF( void )           /*     69 */
{
    /* TODO!!! - don't have version of Windows which has this */
}

void test_SPI_SETSCREENREADER( void )           /*     71 */
{
    /* TODO!!! - don't have version of Windows which has this */
}

/*
 * Registry entries for the system parameters.
 * Names are created by 'SET' flags names.
 * We assume that corresponding 'GET' entries use the same registry keys.
 */
DWORD WINAPI SysParamsThreadFunc( LPVOID lpParam )
{
    test_SPI_SETBEEP();                         /*      1 */
    test_SPI_SETMOUSE();                        /*      4 */
    test_SPI_SETBORDER();                       /*      6 */
    test_SPI_SETKEYBOARDSPEED();                /*     10 */
    test_SPI_ICONHORIZONTALSPACING();           /*     13 */
    test_SPI_SETSCREENSAVETIMEOUT();            /*     14 */
    test_SPI_SETSCREENSAVEACTIVE();             /*     17 */
    test_SPI_SETGRIDGRANULARITY();              /*     19 */
    test_SPI_SETKEYBOARDDELAY();                /*     23 */
    test_SPI_ICONVERTICALSPACING();             /*     24 */
    test_SPI_SETICONTITLEWRAP();                /*     26 */
    test_SPI_SETMENUDROPALIGNMENT();            /*     28 */
    test_SPI_SETDOUBLECLKWIDTH();               /*     29 */
    test_SPI_SETDOUBLECLKHEIGHT();              /*     30 */
    test_SPI_SETDOUBLECLICKTIME();              /*     32 */
    test_SPI_SETMOUSEBUTTONSWAP();              /*     33 */
    test_SPI_SETFASTTASKSWITCH();               /*     36 */
    test_SPI_SETDRAGFULLWINDOWS();              /*     37 */
    test_SPI_SETWORKAREA();                     /*     47 */
    test_SPI_SETSHOWSOUNDS();                   /*     57 */
    test_SPI_SETKEYBOARDPREF();                 /*     69 */
    test_SPI_SETSCREENREADER();                 /*     71 */

    end_tests();

    SendMessage( ghTestWnd, WM_DESTROY, 0, 0 );
    return 0;
}

int main( void )
{
    WNDCLASS wc;
    MSG msg;
    HANDLE hThread;
    DWORD dwThreadId;
    HANDLE hInstance = GetModuleHandle( NULL );

    change_counter = 0;
    change_last_param = 0;
    strcpy( change_reg_section, "" );

    wc.lpszClassName = "SysParamsTestClass";
    wc.lpfnWndProc = SysParamsTestWndProc;
    wc.style = CS_OWNDC | CS_VREDRAW | CS_HREDRAW;
    wc.hInstance = hInstance;
    wc.hIcon = LoadIcon( NULL, IDI_APPLICATION );
    wc.hCursor = LoadCursor( NULL, IDC_ARROW );
    wc.hbrBackground = (HBRUSH)(COLOR_WINDOW + 1 );
    wc.lpszMenuName = 0;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    RegisterClass( &wc );

    ghTestWnd = CreateWindow( "SysParamsTestClass", "Test System Parameters Application",
                             WS_OVERLAPPEDWINDOW, 0, 0, 100, 100, NULL, NULL, hInstance, NULL );
    ShowWindow( ghTestWnd, SW_SHOWNORMAL );

    hThread = CreateThread( NULL, 0, SysParamsThreadFunc, 0, 0, &dwThreadId );
   
    if (hThread == NULL)
    {
        printf( "CreateThread failed." );
        exit( 1 );
    }
    CloseHandle( hThread );

    while( GetMessage( &msg, NULL, 0, 0 )) {
        TranslateMessage( &msg );
        DispatchMessage( &msg );
    }

    return msg.wParam;
}
