/* Helper module for testing Win32 implementations.

  Copyright 2002 Andriy Palamarchuk

  wt_helper library guide

    1. Introduction

  Example of usage:

    ok( 1 == 1, "Successful test" );

    START_WINE_TODO;
    ok( 1 == 2, "Failing TODO test" );
    ok( 1 == 2, "Failing TODO test" );
    END_TODO;

    WINE_TODO(ok( 1 == 2, "Failing test" ));

    ok( 1 == 2, "Failing test" );

    end_tests();


    wt_helper library implements functions, useful for tests creation
    in C. The library confirms to the WINE testing conventions.
    See WINE Developer's Guide for details.

    Terms are used in this document:
        test application - the application which is created with using wt_helper
                library and has purpose to test Win32 API implementation.
        test - one condition checking.
    
    2. Basic tests
    Function "ok" is used to test individual conditions. It fails if condition is
    not-zero and reports the failure according to WineTest conventions.

    Usage:

        ok( <condition>, <test description> )

    Example:
        ok( GetSystemMetrics( SM_CXICONSPACING ) >= 32, "Minimal icon spacing" );

    Function "fail" implements test which always fails.

    Usage:
        fail( <description> )

    Call function end_tests in the end of test application.
    Note, this function does not return, exits the application.
    Call it after all processing is done.

    3. TODO tests
    There will be a lot of tests which will fail for WINE and you won't have time
    or skills to fix the problems immediately. Submit these tests anyway, but mark
    them as TODO.
    TODO tests are tests which are known to fail and should be fixed.
    Tests in TODO section are executed, failures are ignored, test successes are
    reported. Test should be moved out of TODO section if it starts to work.
    TODO sections are conditional, because they are different for different
    Win32 API implementations.
    Obviously, all the tests should succeed for all Microsoft Win32 API
    implementations.

    To mark block of code as TODO section:
    START_TODO( <condition> );
    <TODO tests>
    END_TODO;

    TODO section for WINE implementation:
    START_WINE_TODO;
    <TODO tests>
    END_TODO;

    START_WINE_TODO is the same as START_TODO( get_w32_impl() == W32_WINE ).

    To mark individual tests as TODO you can use macros:
    TODO( <condition>, <test> );
    WINE_TODO( <test> );

    4. WONTFIX tests
    Some discrepancies between Microsoft and other Win32 API implementations are
    intentional. Good example of this situation is when different versions of Windows
    have different behavior, but we decide to implement only, e.g Windows NT
    behaviour, not Windows 95. You can mark such failing tests as WONTFIX.
    WONTFIX tests are tests which are known to fail and are not supposed to be fixed.
    No attempt is made to execute WONTFIX tests.

    You can mark WONTFIX section with:
    START_WONTFIX( <condition> );
    <WONTFIX tests>
    END_WONTFIX;

    WONTFIX section for WINE implementation:
    START_WINE_WONTFIX;
    <WONTFIX tests>
    END_WONTFIX;

    START_WINE_WONTFIX is the same as START_WONTFIX( get_w32_impl() == W32_WINE ).

    To mark individual tests as WONTFIX you can use macros:
    WONTFIX( <condition>, <test> );
    WINE_WONTFIX( <test> );
    

    5. Windows compatibility

    get_ver - returns pointer to record of type OSVERSIONINFO.
        Internally it uses function GetVersionEx to initialize the structure.
        See documentation about type OSVERSIONINFO and function GetVersionEx
        for details.

    has_unicode - returns non-zero value if the underlying platform
        supports Unicode. Test applications are supposed to be compiled
        with undefined UNICODE preprocessor definition. Always call "W"
        variations of Win32 API explicitely.

    6. Test environment

    get_checks_count - returns number of performed checks.

    in_todo - returns not-zero value if called in active TODO section.

    get_w32_impl - current Win32 API implementations the tests are running on.
        The implementation is defined at run-time. The same Windows
        test application binary can run on different Win32 API implementations
        without recompilation.

    7. Library configuration

    Following environment variables affect behaviour of the library:

    WINETEST_DEBUG - level of verbosity of debugging invormation. Level 1 is used
        if the environment variable is not set.
    WINETEST_PLATFORM - Win32 API implementation the test runs on.
        This environment variable overrides automatic platform detection.
        Allowed values: "windows", "wine".
    WINETEST_DIE_ON_FAILURE - number indicator whether tests exit
        on failure. If zero - all tests are performed disregarding whether they fail
        or succeed. Number of failed tests are reported.
        If not-zero number value - test application exists on test failure.
        Value 1 is used if the environment variable is not set.

    Don't set values for these environment variables in production version of the
    test application. These variables are managed by the make script which launches
    tests. However you can change the variables while you debug the tests.
*/

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "wt_helper.h"

#define TRUE 1
#define FALSE 0

/* counter of TODO sections nesting */
int todo_level = 0;

/* indicator of the current active level of TODO sections */
int active_todo = 0;

/* check number */
int checks = 0;

/* number of TODO checks */
int todos = 0;

/* indicator whether test application should exit on failure. */
int die_on_failure = 1;

/* number of failures */
int failures = 0;

/* Win32 API implementation application is running on */
W32_IMPL w32_impl;

#define WINETEST_PLATFORM_WINDOWS       "windows"
#define WINETEST_PLATFORM_WINE          "wine"

/* windows version information */
OSVERSIONINFO version_info;

/* indicator whether underlying platform supports unicode */
int has_unicode_support;

/* verbosity level */
int winetest_debug = 1;

/* indicator the library is initialized */
int wt_helper_initialized = 0;

/* detect Win32 implementation the test application runs on*/
void detect_w32_impl()
{
    HKEY hKey;

    if (RegOpenKey( HKEY_LOCAL_MACHINE, "Software\\Wine\\Wine\\Config",
                    &hKey ) == ERROR_SUCCESS)
    {
        w32_impl = W32_WINE;
        RegCloseKey( hKey );
    }
    else
    {
        w32_impl = W32_MS ;
    }
}

/*
 * The library initialization.
 */
void init_wt_helper( void )
{
    if (!wt_helper_initialized)
    {
        char *env_var;
        wt_helper_initialized = 1;

        env_var = getenv( "WINETEST_DEBUG" );
        if ( env_var != NULL )
            winetest_debug = atoi( env_var );
        env_var = getenv( "WINETEST_DIE_ON_FAILURE" );
        if ( env_var != NULL )
            die_on_failure = atoi( env_var );

        env_var = getenv( "WINETEST_PLATFORM" );
        if (env_var == NULL)
        {
            detect_w32_impl();
        }
        else if (!strcmp( WINETEST_PLATFORM_WINDOWS, env_var ))
        {
            w32_impl = W32_MS;
        }
        else if (!strcmp( WINETEST_PLATFORM_WINE, env_var ))
        {
            w32_impl = W32_WINE;
        }
        else
        {
            detect_w32_impl();
        }

        version_info.dwOSVersionInfoSize = sizeof (version_info);
        GetVersionEx( &version_info );

        has_unicode_support =
            (get_ver()->dwPlatformId == VER_PLATFORM_WIN32_NT) &&
            (get_ver()->dwMajorVersion >= 4);
    }
}

/*
 * Returns Win32 API implementation application is running on
 */
W32_IMPL get_w32_impl( void )
{
    init_wt_helper();
    return w32_impl;
}

/*
 * Returns windows version information.
 * See documentation for OSVERSIONINFO and GetVersionEx for details.
 */
OSVERSIONINFO *get_ver( void )
{
    init_wt_helper();
    return &version_info;
}

/*
 * Returns non-zero value if the underlying platform supports Unicode.
 */
int has_unicode( void )
{
    init_wt_helper();
    return has_unicode_support;
}

/*
 * Returns number of performed checks.
 */
int get_checks_count( void )
{
    return checks;
}

/*
 * Internally used function for processing test failure.
 * Currently prints message and exists with error code 1.
 *   - msg - message to print
 *   - file - test application source code file name
 *   - line - test application source code file line number
 */
void process_failure( char *msg, char *file, int line )
{
    
    init_wt_helper();
    fprintf( stderr, "%s\ncheck #%i, file %s, line %i\n",
             msg, checks, file, line );
    failures++;
    if (die_on_failure)
    {
        ExitProcess( 1 );
    }
}

/*
 * Returns not-zero value when is called in active TODO section.
 */
int in_todo()
{
    return active_todo;
}

/*
 * Starts TODO section. Not thread-safe.
 * params:
 *   - condition - TODO condition;
 */
void _start_todo( int condition, char *file, int line )
{
    todo_level++;
    if (condition)
    {
        if (active_todo != 0)
        {
            process_failure( "Active TODO sections can't be nested",
                             file, line );
        }
        if (todo_level == 0)
        {
            process_failure( "Incorrect nesting of TODO sections",
                             file, line );
        }
        
        active_todo = todo_level;
    }
}

/*
 * Ends TODO section.
 */
void _end_todo( char *file, int line )
{
    todo_level--;
    if (todo_level < 0)
    {
        process_failure(
            "Incorrect processing of TODO nestings - negative TODO level",
            file, line );
    }
    
    if ((active_todo != 0) && (todo_level == active_todo))
    {
        active_todo = 0;
    }
}

/*
 * Checks condition.
 * Parameters:
 *   - condition - condition to check;
 *   - msg test description;
 *   - file - test application source code file name of the check
 *   - line - test application source code file line number of the check
 */
void _ok( int condition, char *msg, char *file, int line )
{
    checks++;
    if (active_todo != 0)
    {
        todos++;
    }

    if (!condition && (active_todo == 0))
    {
        process_failure( msg, file, line );
    }

    if (condition && (active_todo != 0))
    {
        char *todo_test_succeded =
            "CONGRATULATIONS!!! Your TODO test succeded. "
            "Move it out of the TODO section";
        char *s = malloc( strlen( todo_test_succeded ) + strlen( msg ) + 15 );
        sprintf( s, "%s\n%s", todo_test_succeded, msg );
        process_failure( s, file, line );
        free( s );
    }
}

/*
 * Prints test statistics. Cleans test environment.
 * Does not return, exits the application
 */
void end_tests( void )
{
    init_wt_helper();
    if (winetest_debug)
    {
        printf( "Tests completed.\n" );
        printf( "Performed %d checks", checks );
        if (todos)
        {
            printf( ", %d from them (%d%%) are TODO",
                    todos, todos * 100 / checks );
        }
        if (failures)
        {
            printf( ", %d from them (%d%%) are failures",
                    failures, failures * 100 / checks );
        }

        printf( "\n" );
    }
    ExitProcess( failures ? 1 : 0 );
}
