/*
 * Wine Conformance Test EXE
 *
 * Copyright 2003 Jakob Eriksson   (for Solid Form Sweden AB)
 * Copyright 2003 Dimitrie O. Paun
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 *
 * This program is dedicated to Anna Lindh,
 * Swedish Minister of Foreign Affairs.
 * Anna was murdered September 11, 2003.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>

#include <wtypes.h>
#include <shellapi.h>
#include <shellapi.h>

#include "winetests.h"

#define MARKER "@RESULTS@"

void fatal(const char* msg)
{
    MessageBox( NULL, msg, "Error", MB_ICONERROR | MB_OK );
    exit(1);
}

char *strmake(const char *fmt, ...)
{
    int n;
    size_t size = 100;
    char *p;
    va_list ap;

    if (!(p = malloc (size))) fatal("Out of memory.");
    

    while (1)
    {
        va_start(ap, fmt);
	n = vsnprintf (p, size, fmt, ap);
	va_end(ap);
        if (n > -1 && n < size) return p;
	size = min( size*2, n+1 );
	if(!(p = realloc (p, size))) fatal("Out of memory.");
    }
}

int print_version(FILE *fout)
{
    OSVERSIONINFOEX ver;

    ver.dwOSVersionInfoSize = sizeof(OSVERSIONINFOEX); 
    if (!GetVersionEx ((OSVERSIONINFO *) &ver))
    {
	ver.dwOSVersionInfoSize = sizeof(OSVERSIONINFO); 
	if (!GetVersionEx ((OSVERSIONINFO *) &ver))
	    return 0;
    }

    fprintf(fout, "    dwMajorVersion=%ld\n    dwMajorVersion=%ld\n"
		  "    dwBuildNumber=%ld\n    PlatformId=%ld\n    szCSDVersion=%s\n",
	          ver.dwMajorVersion, ver.dwMinorVersion, ver.dwBuildNumber, 
		  ver.dwPlatformId, ver.szCSDVersion);

    if (ver.dwOSVersionInfoSize < sizeof(OSVERSIONINFOEX)) return 1;
    
    fprintf(fout, "    wServicePackMajor=%d\n    wServicePackMinor=%d\n"
		  "    wSuiteMask=%d\n    wProductType=%d\n    wReserved=%d\n",
		  ver.wServicePackMajor, ver.wServicePackMinor, ver.wSuiteMask, 
		  ver.wProductType, ver.wReserved);
    return 1;
}

void deletePath( char *path )
{
    SHFILEOPSTRUCT fileop;

    path[strlen( path ) + 1] = '\0';

    fileop.hwnd = NULL;
    fileop.wFunc = FO_DELETE;
    fileop.pFrom = path;
    fileop.pTo = NULL;
    fileop.fFlags = FOF_NOCONFIRMATION | FOF_NOERRORUI;

    SHFileOperation( &fileop );
}

int count_tests()
{
    struct wine_test* test;
    int total = 0;

    for (test = wine_tests; test->name; test++)
	total += test->subtest_count;

    return total;
}

LPVOID extract_rcdata (const char *name, DWORD* size)
{
    HRSRC rsrc;
    HGLOBAL hdl;
    
    if (!(rsrc = FindResource( 0, name, RT_RCDATA ))) return 0;
    if (!(*size = SizeofResource( 0, rsrc ))) return 0;
    if (!(hdl = LoadResource( 0, rsrc ))) return 0;
    return LockResource(hdl);
}

int extract_test (const char *dir, struct wine_test* test)
{
    BYTE* code;
    DWORD size;
    FILE* fout;

    if (!(code = extract_rcdata(test->name, &size))) return 0;
    
    test->is_elf = (code[1] == 'E' && code[2] == 'L' && code[3] == 'F');
    test->exename = strmake("%s/%s.exe%s", dir, test->name,
			    test->is_elf ? ".so" : "");

    if((fout = fopen(test->exename, "wb")))
    {
        fwrite(code, size, 1, fout);
        fclose(fout);
    }
    return fout != 0;
}

int run_test(FILE* logfp, struct wine_test* test, const char* subtest)
{
    char line[512], *cmd;
    FILE *fp;

    fprintf( logfp, "%s:%s start\n", test->name, subtest );
   
    cmd = strmake("%s %s %s", test->is_elf ? "wine " : "", test->exename, subtest);
    if ((fp = popen( cmd, "r" )))
    {
	while (fgets( line, sizeof(line), fp ))
	    fprintf( logfp, "%s", line );
	fclose( fp );
    }
    else fprintf( logfp, "Failed to execute\n");

    fprintf( logfp, "%s:%s done\n", test->name, subtest );

    return fp != 0;
}

int WINAPI WinMain( HINSTANCE hInst, HINSTANCE hPrevInst, LPSTR cmdline, int cmdshow )
{
    struct wine_test* test;
    int nr_of_tests, subtest;
    char *tempdir, *logfile;
    char *html, *marker;
    DWORD htmlSize;
    FILE *fp;

    SetErrorMode( SEM_FAILCRITICALERRORS );

//    if(MessageBox( 0, " This progam will run a series of conformance tests for Windows.\n"
//	    	  " Proceed with tests? ", "Wine Conformance Tester", 
//		  MB_ICONQUESTION | MB_YESNO ) != IDYES) return 0;

    printf( "Setting up environment...\n" );
    if (!(tempdir = tempnam(0, "wct"))) fatal("Can't create temporary dir.");
    printf( "    tempdir=%s\n", tempdir );
    if (mkdir( tempdir, 0777 )) fatal(strmake("Could not create directory: %s", tempdir));
    if (chdir( tempdir )) fatal(strmake("Could not make %s current directory.", tempdir));

    if (!(logfile = tempnam(0, "res"))) fatal("Can't create result file.");
    logfile = strmake("%s.html", logfile );
    printf( "    logfile=%s\n", logfile );
 
    if (!(html = extract_rcdata("results", &htmlSize))) fatal("Can't find HTML.");
    //html[htmlSize - 1] = 0; /* make sure the HTML is ended properly */
    if (!(marker = strstr(html, MARKER))) fatal("Invalid HTML."); 

    /* Init log file */
    if (!(fp = fopen( logfile, "w" ))) fatal("Could not open log file.");

    fwrite( html, marker - html, 1, fp);

    fprintf( fp, "Tests from build %s\n", "FIXME" ); 
    
    fprintf( fp, "Operating system version:\n");
    if (!print_version(fp)) fprintf( fp, "    N/A\n");
 
    fprintf( fp, "Test output:\n" );

    printf( "Extracting tests...\n" );
    for(test = wine_tests; test->name; test++)
	if (!extract_test( tempdir, test )) 
	    fatal(strmake("Failed to extract %s", test->name));

    nr_of_tests = count_tests();

    printf( "Running the %d tests...\n", nr_of_tests );
    for(test = wine_tests; test->name; test++)
	for (subtest = 0; subtest < test->subtest_count; subtest++)
	    run_test( fp, test, test->subtests[subtest] );

    marker += strlen(MARKER);
    fwrite( marker, htmlSize - (marker - html), 1, fp);
    fclose( fp );

    printf( "Cleaning up...\n");
    deletePath( tempdir );

    printf( "Starting web browser...\n");
    ShellExecute( NULL, "open", logfile, NULL, NULL, SW_SHOWNORMAL );

    printf("Done!\n");
    return 0;
}

