#ifndef WINE_WINE_IPSHARED_H_
#define WINE_WINE_IPSHARED_H_

#include "windef.h"

#define MAX_INTERFACE_PHYSADDR    8
#define MAX_INTERFACE_DESCRIPTION 256

#ifdef __cplusplus
extern "C" {
#endif

/* This module implements functions shared by DLLs that need to enumerate
 * network interfaces and get interface statistics.  It's meant to hide
 * some problematic defines like socket(), as well as provide only one file
 * that needs to be ported to implement these functions on different platforms,
 * since the Windows API provides multiple ways to get at this info.
 *
 * That said, I hacked it together in a hurry, so if you don't like the
 * abstraction or you don't like me feel free to dropkick it.
 */

DWORD getNumInterfaces(void);
DWORD getNumNonLoopbackInterfaces(void);

/* Fills indexes with all known indexes; numIndexes is the number of DWORDS
 * in indexes.  You can find out ahead of time how many you'll need by calling
 * getNumInterfaces(), assuming the count doesn't change between a call to it
 * and a call to this.
 * Returns 0 on success, -1 on failure.
 */
int getInterfaceIndexes(PDWORD indexes, int numIndexes);

/* Like getInterfaceIndexes, but filters out loopback interfaces.
 * numIndexes can be determined by calling getNumNonLoopbackInterfaces().
 */
int getNonLoopbackInterfaceIndexes(PDWORD indexes, int numIndexes);

/* ByName/ByIndex versions of various getter functions. */

// can be used as quick check to see if you've got a valid index, returns NULL
// if not.  The buffer's only valid till the next call, so copy it right away
// if you care.
const char *getInterfaceNameByIndex(DWORD index);

DWORD getInterfaceIndexByName(const char *name);

/* This bunch returns IP addresses, and INADDR_ANY or INADDR_NONE if not found,
 * appropriately depending on the f/n.
 */
DWORD getInterfaceIPAddrByName(const char *name);
DWORD getInterfaceIPAddrByIndex(DWORD index);
DWORD getInterfaceMaskByName(const char *name);
DWORD getInterfaceMaskByIndex(DWORD index);
DWORD getInterfaceBCastAddrByName(const char *name);
DWORD getInterfaceBCastAddrByIndex(DWORD index);

/* Gets a few physical charactersistics of a device:  MAC addr len, MAC addr,
 * and type as one of the MIB_IF_TYPEs.
 * len's in-out: on in, needs to say how many bytes are available in addr,
 * and should be at least MAX_INTERFACE_PHYSADDR.  On out, it's how many bytes
 * were set.
 * Returns ERROR_INVALID_PARAMETER if name, len, addr, or type is NULL.
 * Returns ERROR_INVALID_DATA if name/index isn't valid.
 * May return other errors, e.g. ERROR_OUTOFMEMORY or ERROR_NO_MORE_FILES,
 * if internal errors occur.
 * Returns NO_ERROR on success.
 */
DWORD getInterfacePhysicalByName(const char *name, PDWORD len, PBYTE addr,
 PDWORD type);
DWORD getInterfacePhysicalByIndex(DWORD index, PDWORD len, PBYTE addr,
 PDWORD type);

/* Get the operational status as a (MIB_)IF_OPER_STATUS type.
 */
DWORD getInterfaceStatusByName(const char *name, PDWORD status);
DWORD getInterfaceStatusByIndex(DWORD index, PDWORD status);

DWORD getInterfaceMtuByName(const char *name, PDWORD mtu);
DWORD getInterfaceMtuByIndex(DWORD index, PDWORD mtu);

/* This struct is bitwise-identical to the portion of MIB_IFROW after wszName,
 * and to the private (to WINE) IFEntry in wsock32/wscontrol.h.  This is
 * deliberate--both wscontrol and iphlpapi's interface entry getting can be
 * implemented using this.
 */
typedef struct _InterfaceEntry
{
  DWORD index;
  DWORD type;
  DWORD mtu;
  DWORD speed;
  DWORD physAddrLen;
  BYTE  physAddr[MAX_INTERFACE_PHYSADDR];
  DWORD adminStatus;
  DWORD operStatus;
  DWORD lastChange;
  DWORD inOctets;
  DWORD inUcastPkts;
  DWORD inNUcastPkts;
  DWORD inDiscards;
  DWORD inErrors;
  DWORD inUnknownProtos;
  DWORD outOctets;
  DWORD outUcastPkts;
  DWORD outNUcastPkts;
  DWORD outDiscards;
  DWORD outErrors;
  DWORD outQLen;
  DWORD descrLen;
  BYTE  descr[MAX_INTERFACE_DESCRIPTION];
} InterfaceEntry;

/* Big hummer--fills in the InterfaceEntry by name/index.
 * Returns ERROR_INVALID_PARAMETER if name or entry is NULL, ERROR_INVALID_DATA
 * if name/index isn't valid, and NO_ERROR otherwise.
 */
DWORD getInterfaceEntryByName(const char *name, InterfaceEntry *entry);
DWORD getInterfaceEntryByIndex(DWORD index, InterfaceEntry *entry);

/* Converts the network-order bytes in addr to a printable string.  Returns
 * string.
 */
char *toIPAddressString(unsigned int addr, char string[16]);

/* Statistics-related functions */

/* Bitwise-equivalent to MIBICMPSTATS */
typedef struct _ICMPStats
{
  DWORD msgs;
  DWORD errors;
  DWORD destUnreachs;
  DWORD timeExcds;
  DWORD parmProbs;
  DWORD srcQuenchs;
  DWORD redirects;
  DWORD echos;
  DWORD echoReps;
  DWORD timestamps;
  DWORD timestampReps;
  DWORD addrMasks;
  DWORD addrMaskReps;
} ICMPStats;

typedef struct _ICMPInfo
{
  ICMPStats inStats;
  ICMPStats outStats;
} ICMPInfo;

/* Gets ICMP statistics into stats.  Returns ERROR_INVALID_PARAMETER if stats is
 * NULL, NO_ERROR otherwise.
 */
DWORD getICMPStats(ICMPInfo *stats);

/* This struct is identical both to MIB_IPSTATS, and to the private (to WINE)
 * IPSNMPInfo struct used by wsock32 (see wscontrol.h).  This is deliberate--
 * both iphlpapi and wsock32 IP stats getting can be implemented using this
 * type.
 */
typedef struct _IPStats
{
  DWORD forwarding;
  DWORD defaultTTL;
  DWORD inReceives;
  DWORD inHdrErrors;
  DWORD inAddrErrors;
  DWORD forwDatagrams;
  DWORD inUnknownProtos;
  DWORD inDiscards;
  DWORD inDelivers;
  DWORD outRequests;
  DWORD routingDiscards;
  DWORD outDiscards;
  DWORD outNoRoutes;
  DWORD reasmTimeout;
  DWORD reasmReqds;
  DWORD reasmOks;
  DWORD reasmFails;
  DWORD fragOks;
  DWORD fragFails;
  DWORD fragCreates;
  DWORD numIf;
  DWORD numAddr;
  DWORD numRoutes;
} IPStats;

/* Gets IP statistics into stats.  Returns ERROR_INVALID_PARAMETER if stats is
 * NULL, NO_ERROR otherwise.
 */
DWORD getIPStats(IPStats *stats);

/* Bitwise-equivalent to MIB_TCPSTATS */
typedef struct _TCPStats {
  DWORD rtoAlgorithm;
  DWORD rtoMin;
  DWORD rtoMax;
  DWORD maxConn;
  DWORD activeOpens;
  DWORD passiveOpens;
  DWORD attemptFails;
  DWORD estabResets;
  DWORD currEstab;
  DWORD inSegs;
  DWORD outSegs;
  DWORD retransSegs;
  DWORD inErrs;
  DWORD outRsts;
  DWORD numConns;
} TCPStats;

/* Gets TCP statistics into stats.  Returns ERROR_INVALID_PARAMETER if stats is
 * NULL, NO_ERROR otherwise.
 */
DWORD getTCPStats(TCPStats *stats);

/* Bitwise-equivalent to MIB_UDPSTATS */
typedef struct _UDPStats {
  DWORD inDatagrams;
  DWORD noPorts;
  DWORD inErrors;
  DWORD outDatagrams;
  DWORD numAddrs;
} UDPStats;

/* Gets UDP statistics into stats.  Returns ERROR_INVALID_PARAMETER if stats is
 * NULL, NO_ERROR otherwise.
 */
DWORD getUDPStats(UDPStats *stats);

/* Route table functions */

int getNumRoutes(void);

/* Minimalist route entry, only has the fields I can actually fill in.  How
 * these map to the different windows route data structures is up to you.
 */
typedef struct _RouteEntry {
  DWORD dest;
  DWORD mask;
  DWORD gateway;
  DWORD ifIndex;
  DWORD metric;
} RouteEntry;

typedef struct _RouteTable {
  DWORD numRoutes;
  RouteEntry routes[1];
} RouteTable;

/* Allocates and returns to you the route table, or NULL if it can't allocate
 * enough memory.  free() the returned table.
 */
RouteTable *getRouteTable(void);

#ifdef __cplusplus
}
#endif

#endif /* ndef WINE_WINE_IPSHARED_H_ */
