/* Copyright (C) 2003 Juan Lang
 * Need real license header here.
 *
 * Implementation notes
 * Windows has an adapter index as a key for adapter-related info.  UNIX
 * interfaces tend to use the interface's name instead.  Fortunately, the
 * if_nametoindex and if_indextoname functions are a standard of some sort,
 * and are present at least on Linux, Solaris, and BSD.
 * Windows may rely on an index being cleared in the topmost 8 bits in some
 * APIs; see GetFriendlyIfIndex and the mention of "backward compatible"
 * indexes.  It isn't clear which APIs would fail with non-backward-compatible
 * indexes, and in any event, on UNIX I've seen indexes like 1, 2, 3, etc. so
 * for now I return the UNIX indexes directly.
 *
 * There are three implemened methods for determining the MAC address of an
 * interface:
 * - a specific IOCTL (Linux)
 * - looking in the ARP cache (at least Solaris)
 * - using the sysctl interface (FreeBSD and MacOSX)
 * Solaris and some others have SIOCGENADDR, but I haven't gotten that to work
 * on the Solaris boxes at SourceForge's compile farm, whereas SIOCGARP does.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <net/if_arp.h>
#include <net/route.h>
#include <sys/ioctl.h>

#if HAVE_SYS_SYSCTL_H
#include <sys/sysctl.h>
#endif

#if HAVE_SYS_SOCKIO_H
#include <sys/sockio.h>
#endif

#if HAVE_NET_IF_DL_H
#include <net/if_dl.h>
#endif

#if HAVE_NET_IF_TYPES_H
#include <net/if_types.h>
#endif

#include "winbase.h"
#include "iprtrmib.h"
#include "wine/ipshared.h"

#if HAVE_STRUCT_SOCKADDR_SA_LEN
#define ifreq_len(ifr) \
 max(sizeof(struct ifreq), sizeof((ifr)->ifr_name)+(ifr)->ifr_addr.sa_len)
#else
#define ifreq_len(ifr) sizeof(struct ifreq)
#endif

#ifndef ETH_ALEN
#define ETH_ALEN 6
#endif

#ifndef INADDR_NONE
#define INADDR_NONE UINT32(0xffffffff)
#endif

#define INITIAL_INTERFACES_ASSUMED 4

typedef void (*scanCallback)(struct ifreq *, void *);

/* Scans all interfaces and calls cb with each one and with data.
 * Assumes cb isn't NULL.
 */
static void scanInterfaces(scanCallback cb, void *data)
{
  int fd;

  if (!cb)
    return;

  fd = socket(PF_INET, SOCK_DGRAM, 0);
  if (fd != -1) {
    int ret, ndx;
    struct ifconf ifc;
    caddr_t ifPtr;

    ret = 0;
    ndx = 0;
    memset(&ifc, 0, sizeof(ifc));
    ifc.ifc_buf = NULL;
    // there is no way to know the interface count beforehand,
    // so we need to loop again and again upping our max each time
    // until returned < max
    do {
      ndx = ndx + INITIAL_INTERFACES_ASSUMED;
      if (ifc.ifc_buf)
        free(ifc.ifc_buf);
      ifc.ifc_len = sizeof(struct ifreq) * ndx;
      ifc.ifc_buf = (char *)malloc(ifc.ifc_len);
      ret = ioctl(fd, SIOCGIFCONF, &ifc);
    } while (ret == 0 && ifc.ifc_len == (sizeof(struct ifreq) * ndx));

    ifPtr = ifc.ifc_buf;
    while (ifPtr < ifc.ifc_buf + ifc.ifc_len) {
      cb((struct ifreq *)ifPtr, data);
      ifPtr += ifreq_len((struct ifreq *)ifPtr);
    }
    if (ifc.ifc_buf)
      free(ifc.ifc_buf);
    close(fd);
  }
}

static int isLoopbackInterface(int fd, const char *name)
{
  int ret = 0;

  if (name) {
    struct ifreq ifr;

    strncpy(ifr.ifr_name, name, IFNAMSIZ);
    ifr.ifr_name[IFNAMSIZ] = '\0';
    if (ioctl(fd, SIOCGIFFLAGS, &ifr) == 0)
      ret = ifr.ifr_flags & IFF_LOOPBACK;
  }
  return ret;
}

static DWORD internalGetNumInterfaces(BOOL excludeLoopback)
{
  DWORD numInterfaces = 0;
  int fd;

  fd = socket(PF_INET, SOCK_DGRAM, 0);
  if (fd != -1) {
    int ret, ndx;
    struct ifconf ifc;
    caddr_t ifPtr;

    ret = 0;
    ndx = 0;
    memset(&ifc, 0, sizeof(ifc));
    ifc.ifc_buf = NULL;
    // there is no way to know the interface count beforehand,
    // so we need to loop again and again upping our max each time
    // until returned < max
    do {
      ndx = ndx + INITIAL_INTERFACES_ASSUMED;
      if (ifc.ifc_buf)
        free(ifc.ifc_buf);
      ifc.ifc_len = sizeof(struct ifreq) * ndx;
      ifc.ifc_buf = (char *)malloc(ifc.ifc_len);
      ret = ioctl(fd, SIOCGIFCONF, &ifc);
    } while (ret == 0 && ifc.ifc_len == (sizeof(struct ifreq) * ndx));

    ifPtr = ifc.ifc_buf;
    while (ifPtr < ifc.ifc_buf + ifc.ifc_len) {
      if (excludeLoopback) {
        if (!isLoopbackInterface(fd, ((struct ifreq *)ifPtr)->ifr_name))
          numInterfaces++;
      }
      else
        numInterfaces++;
      ifPtr += ifreq_len((struct ifreq *)ifPtr);
    }
    if (ifc.ifc_buf)
      free(ifc.ifc_buf);
    close(fd);
  }
  return numInterfaces;
}

DWORD getNumInterfaces(void)
{
  return internalGetNumInterfaces(FALSE);
}

DWORD getNumNonLoopbackInterfaces(void)
{
  return internalGetNumInterfaces(TRUE);
}

struct interfaceIndexEnumeratorState {
  PDWORD indexes;
  int numIndexes;
  int numUsed;
};

/* Assumes data is of type interfaceIndexEnumeratorState.  If there's space for
 * another index, stores the index referred to by ifr in
 * data->indexes[data->numUsed].  Always increments data->numUsed.
 */
void interfaceIndexEnumeratorCallback(struct ifreq *ifr, void *data)
{
  if (ifr && data) {
    struct interfaceIndexEnumeratorState *state =
     (struct interfaceIndexEnumeratorState *)data;

    if (state->numUsed < state->numIndexes)
      state->indexes[state->numUsed] = getInterfaceIndexByName(ifr->ifr_name);
    state->numUsed++;
  }
}

int getInterfaceIndexes(PDWORD indexes, int numIndexes)
{
  struct interfaceIndexEnumeratorState state;

  if (!indexes)
    return -1;

  state.indexes = indexes;
  state.numIndexes = numIndexes;
  state.numUsed = 0;
  scanInterfaces(interfaceIndexEnumeratorCallback, &state);
  return state.numUsed <= state.numIndexes ? 0 : -1;
}

void nonLoopbackInterfaceIndexEnumeratorCallback(struct ifreq *ifr, void *data)
{
  if (ifr && data) {
    struct interfaceIndexEnumeratorState *state =
     (struct interfaceIndexEnumeratorState *)data;
    BYTE addr[MAX_INTERFACE_PHYSADDR];
    DWORD len = sizeof(addr), type;

    if (getInterfacePhysicalByName(ifr->ifr_name, &len, addr, &type) ==
     NO_ERROR && type != MIB_IF_TYPE_LOOPBACK) {
      if (state->numUsed < state->numIndexes)
        state->indexes[state->numUsed] = getInterfaceIndexByName(ifr->ifr_name);
      state->numUsed++;
    }
  }
}

int getNonLoopbackInterfaceIndexes(PDWORD indexes, int numIndexes)
{
  struct interfaceIndexEnumeratorState state;

  if (!indexes)
    return -1;

  state.indexes = indexes;
  state.numIndexes = numIndexes;
  state.numUsed = 0;
  scanInterfaces(nonLoopbackInterfaceIndexEnumeratorCallback, &state);
  return state.numUsed <= state.numIndexes ? 0 : -1;
}

const char *getInterfaceNameByIndex(DWORD index)
{
  static char name[IFNAMSIZ];

  return if_indextoname(index, name);
}

DWORD getInterfaceIndexByName(const char *name)
{
  return if_nametoindex(name);
}

DWORD getInterfaceIPAddrByName(const char *name)
{
  DWORD ret = INADDR_ANY;

  if (name) {
    int fd = socket(PF_INET, SOCK_DGRAM, 0);

    if (fd != -1) {
      struct ifreq ifr;

      strncpy(ifr.ifr_name, name, IFNAMSIZ);
      ifr.ifr_name[IFNAMSIZ] = '\0';
      if (ioctl(fd, SIOCGIFADDR, &ifr) == 0)
        memcpy(&ret, ifr.ifr_addr.sa_data + 2, sizeof(DWORD));
      close(fd);
    }
  }
  return ret;
}

DWORD getInterfaceIPAddrByIndex(DWORD index)
{
  DWORD ret;
  const char *name = getInterfaceNameByIndex(index);

  if (name)
    ret = getInterfaceIPAddrByName(name);
  else
    ret = INADDR_ANY;
  return ret;
}

DWORD getInterfaceBCastAddrByName(const char *name)
{
  DWORD ret = INADDR_ANY;

  if (name) {
    int fd = socket(PF_INET, SOCK_DGRAM, 0);

    if (fd != -1) {
      struct ifreq ifr;

      strncpy(ifr.ifr_name, name, IFNAMSIZ);
      ifr.ifr_name[IFNAMSIZ] = '\0';
      if (ioctl(fd, SIOCGIFBRDADDR, &ifr) == 0)
        memcpy(&ret, ifr.ifr_addr.sa_data + 2, sizeof(DWORD));
      close(fd);
    }
  }
  return ret;
}

DWORD getInterfaceBCastAddrByIndex(DWORD index)
{
  DWORD ret;
  const char *name = getInterfaceNameByIndex(index);

  if (name)
    ret = getInterfaceBCastAddrByName(name);
  else
    ret = INADDR_ANY;
  return ret;
}

DWORD getInterfaceMaskByName(const char *name)
{
  DWORD ret = INADDR_NONE;

  if (name) {
    int fd = socket(PF_INET, SOCK_DGRAM, 0);

    if (fd != -1) {
      struct ifreq ifr;

      strncpy(ifr.ifr_name, name, IFNAMSIZ);
      ifr.ifr_name[IFNAMSIZ] = '\0';
      if (ioctl(fd, SIOCGIFNETMASK, &ifr) == 0)
        memcpy(&ret, ifr.ifr_addr.sa_data + 2, sizeof(DWORD));
      close(fd);
    }
  }
  return ret;
}

DWORD getInterfaceMaskByIndex(DWORD index)
{
  DWORD ret;
  const char *name = getInterfaceNameByIndex(index);

  if (name)
    ret = getInterfaceMaskByName(name);
  else
    ret = INADDR_NONE;
  return ret;
}

#if defined (SIOCGIFHWADDR)
DWORD getInterfacePhysicalByName(const char *name, PDWORD len, PBYTE addr,
 PDWORD type)
{
  DWORD ret;
  int fd;

  if (!name || !len || !addr || !type)
    return ERROR_INVALID_PARAMETER;
  if (*len < MAX_INTERFACE_PHYSADDR)
    return ERROR_INVALID_DATA;

  fd = socket(PF_INET, SOCK_DGRAM, 0);
  if (fd != -1) {
    struct ifreq ifr;

    memset(&ifr, 0, sizeof(struct ifreq));
    strncpy(ifr.ifr_name, name, IFNAMSIZ);
    ifr.ifr_name[IFNAMSIZ] = '\0';
    if ((ioctl(fd, SIOCGIFHWADDR, &ifr)))
      ret = ERROR_INVALID_DATA;
    else {
      switch (ifr.ifr_hwaddr.sa_family)
      {
        case ARPHRD_LOOPBACK:
          *len = 0;
          *type = MIB_IF_TYPE_LOOPBACK;
          break;
        case ARPHRD_ETHER:
          *len = ETH_ALEN;
          *type = MIB_IF_TYPE_ETHERNET;
          break;
        case ARPHRD_FDDI:
          *len = ETH_ALEN;
          *type = MIB_IF_TYPE_FDDI;
          break;
        case ARPHRD_IEEE802: // 802.2 Ethernet && Token Ring, guess TR?
          *len = ETH_ALEN;
          *type = MIB_IF_TYPE_TOKENRING;
          break;
        case ARPHRD_IEEE802_TR: // also Token Ring?
          *len = ETH_ALEN;
          *type = MIB_IF_TYPE_TOKENRING;
          break;
        case ARPHRD_SLIP:
          *len = 0;
          *type = MIB_IF_TYPE_SLIP;
          break;
        case ARPHRD_PPP:
          *len = 0;
          *type = MIB_IF_TYPE_PPP;
          break;
        default:
          *len = min(MAX_INTERFACE_PHYSADDR, sizeof(ifr.ifr_hwaddr.sa_data));
          *type = MIB_IF_TYPE_OTHER;
      }
      if (*len > 0)
        memcpy(addr, ifr.ifr_hwaddr.sa_data, *len);
      ret = NO_ERROR;
    }
    close(fd);
  }
  else
    ret = ERROR_NO_MORE_FILES;
  return ret;
}
#elif defined (SIOCGARP)
DWORD getInterfacePhysicalByName(const char *name, PDWORD len, PBYTE addr,
 PDWORD type)
{
  DWORD ret;
  int fd;

  if (!name || !len || !addr || !type)
    return ERROR_INVALID_PARAMETER;
  if (*len < MAX_INTERFACE_PHYSADDR)
    return ERROR_INVALID_DATA;

  fd = socket(PF_INET, SOCK_DGRAM, 0);
  if (fd != -1) {
    if (isLoopbackInterface(fd, name)) {
      *type = MIB_IF_TYPE_LOOPBACK;
      *len = 0;
    }
    else {
      struct arpreq arp;
      struct sockaddr_in *saddr;

      memset(&arp, 0, sizeof(struct arpreq));
      arp.arp_pa.sa_family = AF_INET;
      saddr = (struct sockaddr_in *)&arp; // proto addr is first member of arp
      saddr->sin_family = AF_INET;
      saddr->sin_addr.s_addr = getInterfaceAddrByName(name);
      if ((ioctl(fd, SIOCGARP, &arp)))
        ret = ERROR_INVALID_DATA;
      else {
        // TODO:  heh:  who said it was ethernet?
        *type = MIB_IF_TYPE_ETHERNET;
        *len = ETH_ALEN;
        memcpy(addr, &arp.arp_ha.sa_data[0], *len);
      }
    }
    else
      ret = ERROR_NO_MORE_FILES;
  }
  return ret;
}
#elif defined (HAVE_SYS_SYSCTL_H) && defined (HAVE_NET_IF_DL_H)
DWORD getInterfacePhysicalByName(const char *name, PDWORD len, PBYTE addr,
 PDWORD type)
{
  DWORD ret;
  struct if_msghdr *ifm;
  struct sockaddr_dl *sdl;
  u_char *p, *buf;
  size_t mibLen;
  int mib[] = { CTL_NET, AF_ROUTE, 0, AF_LINK, NET_RT_IFLIST, 0 };
  int found = 0;

  if (!name || !len || !addr || !type)
    return ERROR_INVALID_PARAMETER;
  if (*len < MAX_INTERFACE_PHYSADDR)
    return ERROR_INVALID_DATA;

  if (sysctl(mib, 6, NULL, &len, NULL, 0) < 0)
    return ERROR_NO_MORE_FILES;

  buf = (u_char *)malloc(mibLen);
  if (!buf)
    return ERROR_NOT_ENOUGH_MEMORY;

  if (sysctl(mib, 6, buf, &mibLen, NULL, 0) < 0) {
    free(buf);
    return ERROR_NO_MORE_FILES;
  }
  for (p = buf; !found && p < buf + mibLen; p += ifm->ifm_msglen) {
    ifm = (struct if_msghdr *)p;
    sdl = (struct sockaddr_dl *)(ifm + 1);

    if (ifm->ifm_type != RTM_IFINFO || (ifm->ifm_addrs & RTA_IFP) == 0)
      continue;

    if (sdl->sdl_family != AF_LINK || sdl->sdl_nlen == 0 ||
     memcmp(sdl->sdl_data, name, max(sdl->sdl_nlen, strlen(name))) != 0)
      continue;

    *len = min(MAX_INTERFACE_PHYSADDR, sdl->sdl_alen);
#if defined(HAVE_NET_IF_TYPES_H)
    switch (sdl->sdl_type)
    {
      case IFT_ETHER:
        *type = MIB_IF_TYPE_ETHERNET;
        break;
      case IFT_FDDI:
        *type = MIB_IF_TYPE_FDDI;
        break;
      case IFT_ISO88024: // Token Bus
        *type = MIB_IF_TYPE_TOKENRING;
        break;
      case IFT_ISO88025: // Token Ring
        *type = MIB_IF_TYPE_TOKENRING;
        break;
      case IFT_PPP:
        *type = MIB_IF_TYPE_PPP;
        break;
      case IFT_SLIP:
        *type = MIB_IF_TYPE_SLIP;
        break;
      case IFT_LOOP:
        *type = MIB_IF_TYPE_LOOPBACK;
        break;
      default:
        *type = MIB_IF_TYPE_OTHER;
    }
#else
    // default if we don't know
    *type = MIB_IF_TYPE_ETHERNET;
#endif
    memcpy(addr, LLADDR(sdl), *len);
    found = 1;
  }
  if (!found)
    ret = ERROR_INVALID_DATA;
  else
    ret = NO_ERROR;
  free(buf);
  return ret;
}
#endif

DWORD getInterfacePhysicalByIndex(DWORD index, PDWORD len, PBYTE addr,
 PDWORD type)
{
  const char *name = getInterfaceNameByIndex(index);

  if (name)
    return getInterfacePhysicalByName(name, len, addr, type);
  else
    return ERROR_INVALID_DATA;
}

DWORD getInterfaceMtuByName(const char *name, PDWORD mtu)
{
  DWORD ret;
  int fd;

  if (!name)
    return ERROR_INVALID_PARAMETER;
  if (!mtu)
    return ERROR_INVALID_PARAMETER;

  fd = socket(PF_INET, SOCK_DGRAM, 0);
  if (fd != -1) {
    struct ifreq ifr;

    strncpy(ifr.ifr_name, name, IFNAMSIZ);
    ifr.ifr_name[IFNAMSIZ] = '\0';
    if ((ioctl(fd, SIOCGIFMTU, &ifr)))
      ret = ERROR_FILE_NOT_FOUND;
    else {
      *mtu = ifr.ifr_mtu;
      ret = NO_ERROR;
    }
  }
  else
    ret = ERROR_NO_MORE_FILES;
  return ret;
}

DWORD getInterfaceMtuByIndex(DWORD index, PDWORD mtu)
{
  const char *name = getInterfaceNameByIndex(index);

  if (name)
    return getInterfaceMtuByName(name, mtu);
  else
    return ERROR_INVALID_DATA;
}

DWORD getInterfaceStatusByName(const char *name, PDWORD status)
{
  DWORD ret;
  int fd;

  if (!name)
    return ERROR_INVALID_PARAMETER;
  if (!status)
    return ERROR_INVALID_PARAMETER;

  fd = socket(PF_INET, SOCK_DGRAM, 0);
  if (fd != -1) {
    struct ifreq ifr;

    strncpy(ifr.ifr_name, name, IFNAMSIZ);
    ifr.ifr_name[IFNAMSIZ] = '\0';
    if ((ioctl(fd, SIOCGIFFLAGS, &ifr)))
      ret = ERROR_FILE_NOT_FOUND;
    else {
      if (ifr.ifr_flags & IFF_UP)
        *status = IF_OPER_STATUS_OPERATIONAL;
      else
        *status = IF_OPER_STATUS_NON_OPERATIONAL;
      ret = NO_ERROR;
    }
  }
  else
    ret = ERROR_NO_MORE_FILES;
  return ret;
}

DWORD getInterfaceStatusByIndex(DWORD index, PDWORD status)
{
  const char *name = getInterfaceNameByIndex(index);

  if (name)
    return getInterfaceStatusByName(name, status);
  else
    return ERROR_INVALID_DATA;
}

DWORD getInterfaceEntryByName(const char *name, InterfaceEntry *entry)
{
  BYTE addr[MAX_INTERFACE_PHYSADDR];
  DWORD ret, len = sizeof(addr), type;

  if (!name)
    return ERROR_INVALID_PARAMETER;
  if (!entry)
    return ERROR_INVALID_PARAMETER;

  if (getInterfacePhysicalByName(name, &len, addr, &type) == NO_ERROR) {
    int nameLen = strlen(name);
    FILE *fp;

    memset(entry, 0, sizeof(InterfaceEntry));
    entry->index = getInterfaceIndexByName(name);
    entry->physAddrLen = len;
    memcpy(entry->physAddr, addr, len);
    entry->type = type;
    // FIXME: how to calculate real speed?
    getInterfaceMtuByName(name, &entry->mtu);
    // lie, there's no "administratively down" here
    entry->adminStatus = MIB_IF_ADMIN_STATUS_UP;
    getInterfaceStatusByName(name, &entry->operStatus);
    // punt on dwLastChange?

    // get interface stats from /proc/net/dev, no error if can't
    // no inUnknownProtos, outNUcastPkts, outQLen
    fp = fopen("/proc/net/dev", "r");
    if (fp) {
      char buf[512] = { 0 }, *ptr;
      int nameFound = 0;

      ptr = fgets(buf, sizeof(buf), fp);
      while (ptr && !nameFound) {
        while (*ptr && isspace(*ptr))
          ptr++;
        if (strncasecmp(ptr, name, nameLen) == 0 && *(ptr + nameLen) == ':')
          nameFound = 1;
        else
          ptr = fgets(buf, sizeof(buf), fp);
      }
      if (nameFound) {
        char *endPtr;

        ptr += nameLen + 1;
        if (ptr && *ptr) {
          entry->inOctets = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          entry->inUcastPkts = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          entry->inErrors = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          entry->inDiscards = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          strtoul(ptr, &endPtr, 10); // skip
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          strtoul(ptr, &endPtr, 10); // skip
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          strtoul(ptr, &endPtr, 10); // skip
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          entry->inNUcastPkts = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          entry->outOctets = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          entry->outUcastPkts = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          entry->outErrors = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          entry->outDiscards = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
      }
      fclose(fp);
    }
    ret = NO_ERROR;
  }
  else
    ret = ERROR_INVALID_DATA;
  return ret;
}

DWORD getInterfaceEntryByIndex(DWORD index, InterfaceEntry *entry)
{
  const char *name = getInterfaceNameByIndex(index);

  if (name)
    return getInterfaceEntryByName(name, entry);
  else
    return ERROR_INVALID_DATA;
}

char *toIPAddressString(unsigned int addr, char string[16])
{
  if (string) {
    struct in_addr iAddr;

    iAddr.s_addr = addr;
    // extra-anal, just to make auditors happy
    strncpy(string, inet_ntoa(iAddr), sizeof(string));
    string[16] = '\0';
  }
  return string;
}

DWORD getICMPStats(ICMPInfo *stats)
{
  FILE *fp;

  if (!stats)
    return ERROR_INVALID_PARAMETER;

  memset(stats, 0, sizeof(ICMPInfo));
  // get most of these stats from /proc/net/snmp, no error if can't
  fp = fopen("/proc/net/snmp", "r");
  if (fp) {
    const char hdr[] = "Icmp:";
    char buf[512] = { 0 }, *ptr;

    do {
      ptr = fgets(buf, sizeof(buf), fp);
    } while (ptr && strncasecmp(buf, hdr, sizeof(hdr) - 1));
    if (ptr) {
      // last line was a header, get another
      ptr = fgets(buf, sizeof(buf), fp);
      if (ptr && strncasecmp(buf, hdr, sizeof(hdr) - 1) == 0) {
        char *endPtr;

        ptr += sizeof(hdr);
        if (ptr && *ptr) {
          stats->inStats.msgs = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->inStats.errors = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->inStats.destUnreachs = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->inStats.timeExcds = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->inStats.parmProbs = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->inStats.srcQuenchs = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->inStats.redirects = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->inStats.echoReps = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->inStats.timestamps = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->inStats.timestampReps = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->inStats.addrMasks = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->inStats.addrMaskReps = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->outStats.msgs = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->outStats.errors = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->outStats.destUnreachs = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->outStats.timeExcds = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->outStats.parmProbs = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->outStats.srcQuenchs = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->outStats.redirects = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->outStats.echoReps = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->outStats.timestamps = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->outStats.timestampReps = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->outStats.addrMasks = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->outStats.addrMaskReps = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
      }
    }
    fclose(fp);
  }
  return NO_ERROR;
}

DWORD getIPStats(IPStats *stats)
{
  FILE *fp;

  if (!stats)
    return ERROR_INVALID_PARAMETER;

  memset(stats, 0, sizeof(IPStats));
  stats->numIf = stats->numAddr = getNumInterfaces();
  stats->numRoutes = getNumRoutes();

  // get most of these stats from /proc/net/snmp, no error if can't
  fp = fopen("/proc/net/snmp", "r");
  if (fp) {
    const char hdr[] = "Ip:";
    char buf[512] = { 0 }, *ptr;

    do {
      ptr = fgets(buf, sizeof(buf), fp);
    } while (ptr && strncasecmp(buf, hdr, sizeof(hdr) - 1));
    if (ptr) {
      // last line was a header, get another
      ptr = fgets(buf, sizeof(buf), fp);
      if (ptr && strncasecmp(buf, hdr, sizeof(hdr) - 1) == 0) {
        char *endPtr;

        ptr += sizeof(hdr);
        if (ptr && *ptr) {
          stats->forwarding = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->defaultTTL = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->inReceives = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->inHdrErrors = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->inAddrErrors = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->forwDatagrams = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->inUnknownProtos = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->inDiscards = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->inDelivers = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->outRequests = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->outDiscards = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->outNoRoutes = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->reasmTimeout = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->reasmReqds = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->reasmOks = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->reasmFails = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->fragOks = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->fragFails = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->fragCreates = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        // hmm, no routingDiscards
      }
    }
    fclose(fp);
  }
  return NO_ERROR;
}

DWORD getTCPStats(TCPStats *stats)
{
  FILE *fp;

  if (!stats)
    return ERROR_INVALID_PARAMETER;

  memset(stats, 0, sizeof(TCPStats));

  // get from /proc/net/snmp, no error if can't
  fp = fopen("/proc/net/snmp", "r");
  if (fp) {
    const char hdr[] = "Tcp:";
    char buf[512] = { 0 }, *ptr;


    do {
      ptr = fgets(buf, sizeof(buf), fp);
    } while (ptr && strncasecmp(buf, hdr, sizeof(hdr) - 1));
    if (ptr) {
      // last line was a header, get another
      ptr = fgets(buf, sizeof(buf), fp);
      if (ptr && strncasecmp(buf, hdr, sizeof(hdr) - 1) == 0) {
        char *endPtr;

        ptr += sizeof(hdr);
        if (ptr && *ptr) {
          stats->rtoAlgorithm = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->rtoMin = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->rtoMin = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->maxConn = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->activeOpens = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->passiveOpens = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->attemptFails = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->estabResets = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->currEstab = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->inSegs = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->outSegs = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->retransSegs = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->inErrs = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->outRsts = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        // hmm, no numConns; it isn't 0 on Windows, either, not sure what it
        // means
      }
    }
    fclose(fp);
  }
  return NO_ERROR;
}

DWORD getUDPStats(UDPStats *stats)
{
  FILE *fp;

  if (!stats)
    return ERROR_INVALID_PARAMETER;

  memset(stats, 0, sizeof(UDPStats));

  // get from /proc/net/snmp, no error if can't
  fp = fopen("/proc/net/snmp", "r");
  if (fp) {
    const char hdr[] = "Udp:";
    char buf[512] = { 0 }, *ptr;


    do {
      ptr = fgets(buf, sizeof(buf), fp);
    } while (ptr && strncasecmp(buf, hdr, sizeof(hdr) - 1));
    if (ptr) {
      // last line was a header, get another
      ptr = fgets(buf, sizeof(buf), fp);
      if (ptr && strncasecmp(buf, hdr, sizeof(hdr) - 1) == 0) {
        char *endPtr;

        ptr += sizeof(hdr);
        if (ptr && *ptr) {
          stats->inDatagrams = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->noPorts = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->inErrors = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->outDatagrams = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
        if (ptr && *ptr) {
          stats->numAddrs = strtoul(ptr, &endPtr, 10);
          ptr = endPtr;
        }
      }
    }
    fclose(fp);
  }
  return NO_ERROR;
}

int getNumRoutes(void)
{
  FILE *fp;
  int numRoutes = 0;

  // get from /proc/net/route, no error if can't
  fp = fopen("/proc/net/route", "r");
  if (fp) {
    char buf[512] = { 0 }, *ptr;


    ptr = fgets(buf, sizeof(buf), fp);
    if (ptr) {
      do {
        ptr = fgets(buf, sizeof(buf), fp);
        if (ptr)
          numRoutes++;
      } while (ptr);
    }
    fclose(fp);
  }
  return numRoutes;
}

RouteTable *getRouteTable(void)
{
  int numRoutes = getNumRoutes();
  RouteTable *ret;

  ret = (RouteTable *)calloc(1, sizeof(RouteTable) +
   (numRoutes - 1) * sizeof(RouteEntry));
  if (ret) {
    FILE *fp;

    // get from /proc/net/route, no error if can't
    fp = fopen("/proc/net/route", "r");
    if (fp) {
      char buf[512] = { 0 }, *ptr;

      ptr = fgets(buf, sizeof(buf), fp);
      if (ptr) {
        do {
          ptr = fgets(buf, sizeof(buf), fp);
          if (ptr) {
            DWORD index;
            char *endPtr;

            while (!isspace(*ptr))
              ptr++;
            *ptr = '\0';
            index = getInterfaceIndexByName(buf);
            if (index) {
              ret->routes[ret->numRoutes].ifIndex = index;
              ptr++;
              if (ptr && *ptr) {
                ret->routes[ret->numRoutes].dest = strtoul(ptr, &endPtr, 16);
                ptr = endPtr;
              }
              if (ptr && *ptr) {
                ret->routes[ret->numRoutes].gateway = strtoul(ptr, &endPtr, 16);
                ptr = endPtr;
              }
              if (ptr && *ptr) {
                strtoul(ptr, &endPtr, 16); // skip
                ptr = endPtr;
              }
              if (ptr && *ptr) {
                strtoul(ptr, &endPtr, 16); // skip
                ptr = endPtr;
              }
              if (ptr && *ptr) {
                strtoul(ptr, &endPtr, 16); // skip
                ptr = endPtr;
              }
              if (ptr && *ptr) {
                ret->routes[ret->numRoutes].metric = strtoul(ptr, &endPtr, 16);
                ptr = endPtr;
              }
              if (ptr && *ptr) {
                ret->routes[ret->numRoutes].mask = strtoul(ptr, &endPtr, 16);
                ptr = endPtr;
              }
            }
            ret->numRoutes++;
          }
        } while (ptr);
      }
      fclose(fp);
    }
  }
  return ret;
}
