// include the basic windows header files and the Direct3D header file
#include <stdio.h>
#include <windows.h>
#include <windowsx.h>
#include <d3d9.h>
#include <d3dx9.h>
#include <assert.h>

// define the screen resolution and keyboard macros
#define SCREEN_WIDTH 800
#define SCREEN_HEIGHT 600
#define KEY_DOWN(vk_code) ((GetAsyncKeyState(vk_code) & 0x8000) ? 1 : 0)
#define KEY_UP(vk_code) ((GetAsyncKeyState(vk_code) & 0x8000) ? 0 : 1)

// include the Direct3D Library files
#pragma comment (lib, "d3d9.lib")
#pragma comment (lib, "d3dx9.lib")

// global declarations
LPDIRECT3D9 d3d;
LPDIRECT3DDEVICE9 d3ddev;
LPD3DXMESH mesh;    // define the mesh pointer
LPD3DXBUFFER adjacency;
//LPDIRECT3DVERTEXBUFFER9 v_buffer = NULL;
//LPDIRECT3DINDEXBUFFER9 i_buffer = NULL;
//LPDIRECT3DTEXTURE9 texture = NULL;

// function prototypes
void initD3D(HWND hWnd);
void render_frame(void);
void cleanD3D(void);
void init_graphics(void);
void init_light(void);

struct CUSTOMVERTEX {FLOAT X, Y, Z; D3DVECTOR NORMAL;};
//#define CUSTOMFVF (D3DFVF_XYZ | D3DFVF_NORMAL)

// the WindowProc function prototype
LRESULT CALLBACK WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);

// the entry point for any Windows program
int WINAPI WinMain(HINSTANCE hInstance,
                   HINSTANCE hPrevInstance,
                   LPSTR lpCmdLine,
                   int nCmdShow)
{
    HWND hWnd;
    WNDCLASSEX wc;

    ZeroMemory(&wc, sizeof(WNDCLASSEX));

    wc.cbSize = sizeof(WNDCLASSEX);
    wc.style = CS_HREDRAW | CS_VREDRAW;
    wc.lpfnWndProc = WindowProc;
    wc.hInstance = hInstance;
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.lpszClassName = "WindowClass";

    RegisterClassEx(&wc);

    hWnd = CreateWindowEx(NULL, "WindowClass", "Our Direct3D Program",
                          WS_OVERLAPPEDWINDOW, 0, 0, SCREEN_WIDTH, SCREEN_HEIGHT,
                          NULL, NULL, hInstance, NULL);

    ShowWindow(hWnd, nCmdShow);

    // set up and initialize Direct3D
    initD3D(hWnd);

    // enter the main loop:

    MSG msg;

    while(TRUE)
    {
        while(PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
        {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
        }

        if(msg.message == WM_QUIT)
            break;

        render_frame();

        if(KEY_DOWN(VK_ESCAPE))
            PostMessage(hWnd, WM_DESTROY, 0, 0);
    }

    // clean up DirectX and COM
    cleanD3D();

    return msg.wParam;
}

// this is the main message handler for the program
LRESULT CALLBACK WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    switch(message)
    {
    case WM_DESTROY:
        {
            PostQuitMessage(0);
            return 0;
        } break;
    }

    return DefWindowProc (hWnd, message, wParam, lParam);
}

// this function initializes and prepares Direct3D for use
void initD3D(HWND hWnd)
{
    d3d = Direct3DCreate9(D3D_SDK_VERSION);

    D3DPRESENT_PARAMETERS d3dpp;

    ZeroMemory(&d3dpp, sizeof(d3dpp));
    d3dpp.Windowed = TRUE;
    d3dpp.SwapEffect = D3DSWAPEFFECT_DISCARD;
    d3dpp.hDeviceWindow = hWnd;
    d3dpp.BackBufferFormat = D3DFMT_X8R8G8B8;
    d3dpp.BackBufferWidth = SCREEN_WIDTH;
    d3dpp.BackBufferHeight = SCREEN_HEIGHT;
    d3dpp.EnableAutoDepthStencil = TRUE;
    d3dpp.AutoDepthStencilFormat = D3DFMT_D16;

    // create a device class using this information and the info from the d3dpp stuct
    d3d->CreateDevice(D3DADAPTER_DEFAULT,
                      D3DDEVTYPE_HAL,
                      hWnd,
                      D3DCREATE_SOFTWARE_VERTEXPROCESSING,
                      &d3dpp,
                      &d3ddev);

    init_graphics();
    init_light();

    d3ddev->SetRenderState(D3DRS_ZENABLE, TRUE);
    d3ddev->SetRenderState(D3DRS_AMBIENT, D3DCOLOR_XRGB(50, 50, 50));

    d3ddev->SetSamplerState(0, D3DSAMP_MAXANISOTROPY, 8);    // anisotropic level
    d3ddev->SetSamplerState(0, D3DSAMP_MINFILTER, D3DTEXF_ANISOTROPIC);    // minification
    d3ddev->SetSamplerState(0, D3DSAMP_MAGFILTER, D3DTEXF_LINEAR);    // magnification
    d3ddev->SetSamplerState(0, D3DSAMP_MIPFILTER, D3DTEXF_LINEAR);    // mipmap
}

// this is the function used to render a single frame
void render_frame(void)
{
    d3ddev->Clear(0, NULL, D3DCLEAR_TARGET, D3DCOLOR_XRGB(0, 0, 0), 1.0f, 0);
    d3ddev->Clear(0, NULL, D3DCLEAR_ZBUFFER, D3DCOLOR_XRGB(0, 0, 0), 1.0f, 0);

    d3ddev->BeginScene();

    // select which vertex format we are using
    //d3ddev->SetFVF(CUSTOMFVF);

    // set the view transform
    D3DXMATRIX matView;
    D3DXMatrixLookAtLH(&matView,
                       &D3DXVECTOR3 (0.0f, 2.0f, 6.0f),    // the camera position
                       &D3DXVECTOR3 (0.0f, 0.0f, 0.0f),      // the look-at position
                       &D3DXVECTOR3 (0.0f, 1.0f, 0.0f));    // the up direction
    d3ddev->SetTransform(D3DTS_VIEW, &matView);

    // set the projection transform
    D3DXMATRIX matProjection;
    D3DXMatrixPerspectiveFovLH(&matProjection,
                               D3DXToRadian(45),
                               (FLOAT)SCREEN_WIDTH / (FLOAT)SCREEN_HEIGHT,
                               1.0f,    // the near view-plane
                               100.0f);    // the far view-plane
    d3ddev->SetTransform(D3DTS_PROJECTION, &matProjection);

    // set the world transform
    static float index = 0.0f; index+=0.03f;
    D3DXMATRIX matRotateY;
    D3DXMatrixRotationY(&matRotateY, index);
    d3ddev->SetTransform(D3DTS_WORLD, &(matRotateY));

    // select the vertex and index buffers to use
    //d3ddev->SetStreamSource(0, v_buffer, 0, sizeof(CUSTOMVERTEX));
    //d3ddev->SetIndices(i_buffer);

    // set the texture
    //d3ddev->SetTexture(0, texture);

    // draw the teapot
    mesh->DrawSubset(0);    
    //d3ddev->DrawIndexedPrimitive(D3DPT_TRIANGLELIST, 0, 0, 24, 0, 12); 

    d3ddev->EndScene(); 

    d3ddev->Present(NULL, NULL, NULL, NULL);

    PostQuitMessage(0);
}

// this is the function that cleans up Direct3D and COM
void cleanD3D(void)
{
    //v_buffer->Release();
    //i_buffer->Release();
    //texture->Release();
    mesh->Release();
    d3ddev->Release();
    d3d->Release();
}

/* http://www.cygnus-software.com/papers/comparingfloats/comparingfloats.htm
 * "I think the answer is 10,000 but since floating point math is imperfect I’ll accept the maxUlps floats above and the maxUlps floats below that value." */
static BOOLEAN AlmostEqual2sComplement(float A, float B, int maxUlps)
{
    int aInt, bInt, intDiff;
    /* Make sure maxUlps is non-negative and small enough that the
     * default NAN won't compare as equal to anything. */
    assert(maxUlps > 0 && maxUlps < 4 * 1024 * 1024);
    aInt= *(int*)&A;
    /* Make aInt lexicographically ordered as a twos-complement int */
    if (aInt < 0)
        aInt = 0x80000000 - aInt;
    /* Make bInt lexicographically ordered as a twos-complement int */
    bInt = *(int*)&B;
    if (bInt < 0)
        bInt = 0x80000000 - bInt;
    intDiff = abs(aInt - bInt);
    if (intDiff <= maxUlps)
        return TRUE;
    if (fabs(A-B)<0.0000000001)
        return TRUE;
    printf("\t\t%g\t%g\t%d\n",A,B,intDiff);
    return FALSE;
}

CUSTOMVERTEX verts[10000];

/* 
 * Adapted from pg. 246, Frank D Luna, "Introduction to 3D Game Programming with Direct X 9.0c: 
 * A Shader Approach (Wordware Game and Graphics Library) [Paperback]," 2006. 
 */
void ComputeNormal(D3DXVECTOR3* p0,
                   D3DXVECTOR3* p1,
                   D3DXVECTOR3* p2,
                   D3DXVECTOR3* out)
{
    D3DXVECTOR3 u = *p1 - *p0;
    D3DXVECTOR3 v = *p2 - *p0;
    
    D3DXVec3Cross(out, &u, &v);
    D3DXVec3Normalize(out, out);
}                   

FLOAT ComputeArea(D3DXVECTOR3* p0,
                  D3DXVECTOR3* p1,
                  D3DXVECTOR3* p2)
{
    D3DXVECTOR3 u = *p1 - *p0;
    D3DXVECTOR3 v = *p2 - *p0;
    D3DXVECTOR3 out;
    
    D3DXVec3Cross(&out, &u, &v);
    return 0.5*D3DXVec3Length(&out);
}                   

D3DXVECTOR3 normals[1000][1000];
FLOAT areas[1000][1000];
int num_normals[1000];

#define TEST 11

// this is the function that puts the 3D models into video RAM
void init_graphics(void)
{
    static const FLOAT radius = 1.0f;
    static const FLOAT length = 1.0f;
    static const UINT slices = 10;
    static const UINT stacks = 20;
    static const FLOAT fNormalEdgeThreshold = -1.01f;
    LPD3DXMESH clone = NULL;

    D3DXCreateCylinder(d3ddev, radius, radius, length, slices, stacks, &mesh, &adjacency);

    if (!SUCCEEDED(D3DXComputeTangentFrameEx(mesh,
                                             D3DX_DEFAULT,
                                             0,
                                             D3DX_DEFAULT,
                                             0,
                                             D3DX_DEFAULT,
                                             0,
                                             D3DDECLUSAGE_NORMAL,
                                             0,
                                             D3DXTANGENT_GENERATE_IN_PLACE | D3DXTANGENT_CALCULATE_NORMALS/* | D3DXTANGENT_WEIGHT_EQUAL*/,
                                             NULL,
                                             -1.01f,
                                             -0.01f,
                                             fNormalEdgeThreshold,
                                             NULL,
                                             NULL)))
    {
        PostQuitMessage(0);
    }

    /*
    if (!SUCCEEDED(D3DXComputeNormals(mesh,NULL)))
    {
        PostQuitMessage(0);
    }
    */

    // vertex buffer
    printf("\nVertex Buffer:\n\n");
    
    IDirect3DVertexBuffer9* vertex_buffer;
    CUSTOMVERTEX* v2;
    DWORD numverts=mesh->GetNumVertices();
    if (SUCCEEDED(mesh->GetVertexBuffer(&vertex_buffer))) 
    {
        CUSTOMVERTEX* vertex;
        if (SUCCEEDED(vertex_buffer->Lock(0,0,(void **)&vertex,D3DLOCK_DISCARD)))
        {
            v2=vertex;
            memcpy(verts,vertex,numverts*sizeof(CUSTOMVERTEX));
            for (int i=0;i<numverts;i++) 
            {
                if (i==TEST)
                {
                    printf("\t%d\t{%g,%g,%g,%g,%g,%g}\n",i,vertex->X,vertex->Y,vertex->Z,vertex->NORMAL.x,vertex->NORMAL.y,vertex->NORMAL.z);
                    /*
                    vertex->NORMAL.x = 0.0f;
                    vertex->NORMAL.y = 0.0f;
                    vertex->NORMAL.z = 0.0f;                
                    */
                }
                vertex++;
            }
            //            vertex_buffer->Unlock();
        }
        //        vertex_buffer->Release();
    }
    
    // index buffer
    printf("\nIndex Buffer:\n\n");

    memset(num_normals, 0, sizeof(num_normals));
    memset(normals, 0, sizeof(normals));
    
    IDirect3DIndexBuffer9* index_buffer;
    DWORD numfaces=mesh->GetNumFaces();
    if (SUCCEEDED(mesh->GetIndexBuffer(&index_buffer))) 
    {
        WORD* index;
        if (SUCCEEDED(index_buffer->Lock(0,0,(void **)&index,D3DLOCK_DISCARD)))
        {
            for (int i=0;i<numfaces;i++) 
            {
                if (index[0]==TEST||index[1]==TEST||index[2]==TEST)
                {
                    printf("\t%d,%d,%d,\n",index[0],index[1],index[2]);
                    D3DXVECTOR3 p0,p1,p2,save;
                    p0.x=v2[index[0]].X;
                    p0.y=v2[index[0]].Y;
                    p0.z=v2[index[0]].Z;
                    printf("\t\t{%g,%g,%g},\n",p0.x,p0.y,p0.z);
                    p1.x=v2[index[1]].X;
                    p1.y=v2[index[1]].Y;
                    p1.z=v2[index[1]].Z;
                    printf("\t\t{%g,%g,%g},\n",p1.x,p1.y,p1.z);
                    p2.x=v2[index[2]].X;
                    p2.y=v2[index[2]].Y;
                    p2.z=v2[index[2]].Z;
                    printf("\t\t{%g,%g,%g},\n",p2.x,p2.y,p2.z);
                    ComputeNormal(&p0,&p1,&p2,&save);
                    printf("\t\t\t{%g,%g,%g},\n",save.x,save.y,save.z);
                    printf("\t\t\tArea = %g,\n",ComputeArea(&p0,&p1,&p2));
                    
                    memcpy(&normals[index[0]][num_normals[index[0]]],&save,sizeof(save));
                    areas[index[0]][num_normals[index[0]]]=ComputeArea(&p0,&p1,&p2);
                    num_normals[index[0]]++;
                    memcpy(&normals[index[1]][num_normals[index[1]]],&save,sizeof(save));
                    areas[index[1]][num_normals[index[1]]]=ComputeArea(&p0,&p1,&p2);
                    num_normals[index[1]]++;
                    memcpy(&normals[index[2]][num_normals[index[2]]],&save,sizeof(save));
                    areas[index[2]][num_normals[index[2]]]=ComputeArea(&p0,&p1,&p2);
                    num_normals[index[2]]++;
                }
                index+=3;
            }
            index_buffer->Unlock();
        }
        index_buffer->Release();
    }

    // average normals
    printf("\nAveraged Normals:\n\n");

    //    for (int i=0;i<1;i++)
    for (int i=TEST;i<TEST+1;i++)
        //    for (int i=0;i<numverts;i++)
    {
        D3DXVECTOR3 avg;
        memset(&avg,0,sizeof(avg));

        /*
        FLOAT dotproduct;
        BOOLEAN dontdo = FALSE;

        for (int j=0;j<num_normals[i];j++)
        {
            for (int k=j+1;k<num_normals[i];k++)
            {
                dotproduct = D3DXVec3Dot(&normals[i][j],&normals[i][k]);
                printf("\t\t%d\t%d\t%d\t{%g,%g,%g}\t{%g,%g,%g}\t%g\n",i,j,k,normals[i][j].x,normals[i][j].y,normals[i][j].z,normals[i][k].x,normals[i][k].y,normals[i][k].z,dotproduct);
                if (dotproduct > fNormalEdgeThreshold)
                {
                    dontdo = TRUE;
                }
            }
        }

        if (dontdo == FALSE)
        {
        */
            for (int j=0;j<num_normals[i];j++)
            {
                printf("\t\t\t%g\t{%g,%g,%g}\n",areas[i][j],normals[i][j].x,normals[i][j].y,normals[i][j].z);
                avg.x+=areas[i][j]*normals[i][j].x;
                avg.y+=areas[i][j]*normals[i][j].y;
                avg.z+=areas[i][j]*normals[i][j].z;
            }
            /*
            avg.x/=num_normals[i];
            avg.y/=num_normals[i];
            avg.z/=num_normals[i];
            */
            D3DXVec3Normalize(&avg,&avg);
            //            AlmostEqual2sComplement(avg.x,v2[i].NORMAL.x,5);
            //            AlmostEqual2sComplement(avg.y,v2[i].NORMAL.y,5);
            //            AlmostEqual2sComplement(avg.z,v2[i].NORMAL.z,5);
            printf("\t%d\t{%g,%g,%g}\t{%g,%g,%g},\n",i,avg.x,avg.y,avg.z,v2[i].NORMAL.x,v2[i].NORMAL.y,v2[i].NORMAL.z);
            /*
        }
            */
    }

    vertex_buffer->Unlock();
    vertex_buffer->Release();

    /*
    // adjacency
    printf("\nAdjacency:\n\n");

    DWORD size = adjacency->GetBufferSize();
    assert( size == numfaces * 3 * sizeof(DWORD) );
    DWORD *ptr = (DWORD *)adjacency->GetBufferPointer();    
    for (int i=0;i<numfaces;i++)
    {
        printf("\t%d,%d,%d,\n",ptr[0],ptr[1],ptr[2]);
        ptr+=3;
    }

    // normals
    //    mesh->CloneMeshFVF(D3DXMESH_MANAGED, 
    //    if (SUCCEEDED(D3DXComputeNormals(mesh,NULL)))
    //    if (SUCCEEDED(D3DXComputeTangentFrameEx(mesh,
                                            D3DX_DEFAULT,0,
                                            D3DX_DEFAULT,0,
                                            D3DX_DEFAULT,0,
                                            D3DDECLUSAGE_NORMAL,0,
                                            D3DXTANGENT_CALCULATE_NORMALS|D3DXTANGENT_GENERATE_IN_PLACE,
                                            NULL,
                                            0.01f,
                                            0.25f,
                                            0.01f,NULL,NULL)))
    //    if (SUCCEEDED(D3DXComputeTangentFrame(mesh,NULL)))
    //    if (SUCCEEDED(D3DXComputeTangentFrame(mesh,D3DXTANGENT_CALCULATE_NORMALS | D3DXTANGENT_GENERATE_IN_PLACE)))
    //    if (SUCCEEDED(D3DXComputeNormals(mesh,(DWORD *)adjacency->GetBufferPointer())))
    {
        printf("\nVertex Buffer after D3DXComputeNormals:\n\n");

        numverts=mesh->GetNumVertices();
        if (SUCCEEDED(mesh->GetVertexBuffer(&vertex_buffer))) 
        {
            CUSTOMVERTEX* vertex;
            CUSTOMVERTEX* v=verts;
            if (SUCCEEDED(vertex_buffer->Lock(0,0,(void **)&vertex,D3DLOCK_DISCARD)))
            {
                for (int i=0;i<numverts;i++) 
                {
                    printf("\t{%g,%g,%g} {%g,%g,%g}\n",
                           vertex->NORMAL.x,vertex->NORMAL.y,vertex->NORMAL.z,
                           v->NORMAL.x,v->NORMAL.y,v->NORMAL.z);
                    //                    printf("\t{%g,%g,%g,%g,%g,%g}\n",vertex->X,vertex->Y,vertex->Z,vertex->NORMAL.x,vertex->NORMAL.y,vertex->NORMAL.z);
                    vertex++;
                    v++;
                }
                vertex_buffer->Unlock();
            }
            vertex_buffer->Release();
        }        
    }
    */
}

// this is the function that sets up the lights and materials
void init_light(void)
{
    D3DLIGHT9 light;    // create the light struct
    D3DMATERIAL9 material;    // create the material struct

    ZeroMemory(&light, sizeof(light));    // clear out the light struct for use
    light.Type = D3DLIGHT_DIRECTIONAL;    // make the light type 'directional light'
    light.Diffuse = D3DXCOLOR(0.5f, 0.5f, 0.5f, 1.0f);    // set the light's color
    light.Direction = D3DXVECTOR3(-1.0f, -0.3f, -1.0f);

    d3ddev->SetLight(0, &light);    // send the light struct properties to light #0
    d3ddev->LightEnable(0, TRUE);    // turn on light #0

    ZeroMemory(&material, sizeof(D3DMATERIAL9));    // clear out the struct for use
    material.Diffuse = D3DXCOLOR(1.0f, 1.0f, 1.0f, 1.0f);    // set diffuse color to white
    material.Ambient = D3DXCOLOR(1.0f, 1.0f, 1.0f, 1.0f);    // set ambient color to white

    d3ddev->SetMaterial(&material);    // set the globably-used material to &material
}
