#!/bin/bash

# Configurable variables.
WINE_DIR=${WINE_DIR:-$PWD}
GECKO_DIR=${GECKO_DIR:-$WINE_DIR/../gecko}
GECKO_MSI=${GECKO_MSI:-wine_gecko-1.4-x86.msi}
BUILD_DIR=${BUILD_DIR:-$WINE_DIR/../.wine-tests}
CLONE_DIR=${CLONE_DIR:-$BUILD_DIR/src}
export CC="${CC:-gcc}"
export CPPFLAGS="${CPPFLAGS:--I/Users/chip/Gentoo/usr/include}"
export CFLAGS="${CFLAGS:--O2}"
export LDFLAGS="${LDFLAGS:--L/Users/chip/Gentoo/usr/lib}"
let JOBS=${JOBS:-$(/usr/sbin/sysctl -n hw.ncpu)+1}

# Options.
CONFIGURE='--x-includes=/opt/X11/include --x-libraries=/opt/X11/lib'
REVISION=HEAD

GECKO_DEST_DIR=$BUILD_DIR/gecko

GETOPT_LONG="/Users/chip/Gentoo/usr/bin/getopt-long"
GIT="/Users/chip/Gentoo/usr/bin/git"

usage()
{
    cat <<__DONE__
usage: $0 [-c|--configure <configure_args>] [-r|--revision <revision>]

Options:
  -u, --update: Does an automatic 'git fetch; git rebase origin' in the Wine
                git repository.
  -c, --configure: Arguments to pass to the configure script in the Wine git
                   repository.
  -r, --revision: Build the tests against this revision of Wine. Can be a
                  tag or a SHA1 commit ID.

Environment variables:
  WINE_DIR: Path to the Wine git repository. If not given, defaults to
            the current directory.
  GECKO_DIR: Path to the Wine Gecko HTML renderer package. If not given,
             defaults to \$WINE_DIR/../gecko.
  GECKO_MSI: The filename of the MSI archive containing this HTML
             renderer. If not given, defaults to wine_gecko-1.2.0-x86.msi.
  BUILD_DIR: Path to put all of $0's temporaries. If not given, defaults to
             \$WINE_DIR/../.wine-tests.
  CLONE_DIR: Path to put a clone of the Wine git repository. If not given,
             defaults to \$BUILD_DIR/src.
  CC: The C compiler to use to build Wine. Defaults to gcc.
  CPPFLAGS: Flags to pass to the C preprocessor. Defaults to something
            sensible for the system on which this script was developed.
  CFLAGS: Flags to pass to the C compiler. Defaults to "-O2".
  LDFLAGS: Flags to pass to the linker. Defaults to something sensible
           for the system on which this script was developed.
  JOBS: Number of simultaneous jobs to run when building Wine. Defaults to
        the number of logical CPUs on this system, plus one.
__DONE__
}

parse_args()
{
    options=`$GETOPT_LONG -o uc:r: -l update,configure:,revision: -- $*`
    err=`$GETOPT_LONG -o uc:r: -l update,configure:,revision: -- $* 2>&1`
    eval set -- $options

    if test $? != 0 -o "$options" != "$err"; then
        usage
        exit 1
    fi

    while test "$1" != ""; do
        case $1 in
            -c|--configure)
                CONFIGURE="$2"
                shift 2
                ;;
            -r|--revision)
                REVISION="$2"
                shift 2
                ;;
            --)
                shift
                break 2
                ;;
            *)
                echo "Unrecognized option '$1'" >&2
                shift
                ;;
        esac
    done
    while test "$1" != ""; do
        #TODO: Set variables specified on the command line.
        echo "Unused argument '$1'" >&2
        shift
    done
}

update_tree()
{
    export GIT_WORK_TREE="$WINE_DIR"
    export GIT_DIR="$GIT_WORK_TREE/.git"
    BRANCH=`$GIT branch | grep "*" | cut -c 3-`
    $GIT fetch; $GIT rebase origin tests
    $GIT checkout "$BRANCH"
    unset GIT_DIR
    unset GIT_WORK_TREE
}

clone_tree()
{
    if test -d "$CLONE_DIR"; then return; fi
    mkdir -p "$BUILD_DIR"
    $GIT clone --reference "$WINE_DIR" --branch tests "$WINE_DIR" "$CLONE_DIR"
    if test "$REVISION" != "HEAD"; then
        pushd "$CLONE_DIR"
        $GIT checkout "$REVISION"
        popd
    fi
    mkdir -p "$GECKO_DEST_DIR"
    cp "$GECKO_DIR/$GECKO_MSI" "$GECKO_DEST_DIR"
}

build_test()
{
    if test -e "$WINE"; then return 0; fi
    cd "$CLONE_DIR"
    echo "Configuring..." && ./configure $CONFIGURE && \
                             make dlls/shell32/Makefile && \
                             sed -i -e 's|sed|/usr/bin/sed|g' \
                                 "$CLONE_DIR/dlls/shell32/Makefile" && \
        echo "Building..." && make -j$JOBS
}

build_failed()
{
    echo "ERROR: Build failed. Suggest manual intervention..."
    exit 1
}

parse_args $@
update_tree
clone_tree
if ! build_test; then
    build_failed
fi

