#!/bin/bash

# Configurable variables.
export WINE_DIR=${WINE_DIR:-$PWD}
export BUILD_DIR=${BUILD_DIR:-$WINE_DIR/../.wine-tests}
export CLONE_DIR=${CLONE_DIR:-$BUILD_DIR/src}
export WINEPREFIX=${WINEPREFIX:-$BUILD_DIR/prefix}
export WINETEST_BIN=${WINETEST_BIN:-$CLONE_DIR/programs/winetest/winetest.exe.so}

# Options.
TIMEOUT=indefinite
TAG="cd-mac10.6"
MAIL="cdavis@mymail.mines.edu"
MODULES=''

export WINE=$CLONE_DIR/wine
export WINELOADER=$WINE
export WINESERVER=$CLONE_DIR/server/wineserver
export SCRIPT_DIR=$PWD/test-scripts
export DYLD_FALLBACK_LIBRARY_PATH="/Users/chip/Gentoo/usr/lib:/opt/X11/lib"
export DYLD_LIBRARY_PATH="/usr/local/lib"

GETOPT_LONG="/Users/chip/Gentoo/usr/bin/getopt-long"

usage()
{
    cat <<__DONE__
usage: $0 [-t|--timeout <seconds>] [-a|--tag <tag>]
          [-m|--mail <email>] [modules...]

Options:
  -t, --timeout: Amount of time to wait before forcibly terminating the test.
                 If not specified, $0 will wait indefinitely.
  -a, --tag: Tag to use to identify report. Defaults to "cd-mac10.6-<options>"
  -m, --mail: Email address used to identify the person submitting the test.
              Can actually be any piece of identifying info. Defaults to
              "cdavis5x" (the IRC nick of the author of this script).
  modules...: Specifies option modules to load when performing tests.

Environment variables:
  WINE_DIR: Path to the Wine git repository. If not given, defaults to
            the current directory.
  BUILD_DIR: Path to put all of $0's temporaries. If not given, defaults to
             \$WINE_DIR/../.wine-tests.
  CLONE_DIR: Path to put a clone of the Wine git repository. If not given,
             defaults to \$BUILD_DIR/src.
  WINEPREFIX: The Wine prefix to use for testing. If not given, defaults to
              \$BUILD_DIR/prefix.
  WINETEST_BIN: Path to the winetest.exe binary. If not given, defaults to
                \$CLONE_DIR/programs/winetest/winetest.exe.so.
  If BUILD_DIR does not exist, $0 will invoke build-wine-test.sh,
  expected to be found in WINE_DIR, to create it for you. Any variables
  accepted by build-wine-test.sh will be passed on automatically.
__DONE__
}

parse_args()
{
    options=`$GETOPT_LONG -o t:a:m: -l timeout:,tag:,mail: -- $*`
    err=`$GETOPT_LONG -o t:a:m: -l timeout:,tag:,mail: -- $* 2>&1`
    eval set -- $options

    if test $? != 0 -o "$options" != "$err"; then
        usage
        exit 1
    fi

    while test "$1" != ""; do
        case $1 in
            -t|--timeout)
                TIMEOUT="$2"
                shift 2
                ;;
            -a|--tag)
                TAG="$2"
                shift 2
                ;;
            -m|--mail)
                MAIL="$2"
                shift 2
                ;;
            --)
                shift
                break 2
                ;;
            *)
                echo "Unrecognized option '$1'" >&2
                shift
                ;;
        esac
    done
    # All others must be module arguments.
    while test "$1" != ""; do
        #TODO: Set variables specified on the command line.
        MODULES="$MODULES $1"
        shift
    done
}


run_test()
{
    cd "$CLONE_DIR"
    export WINE_SCRIPT_DIR=`"$WINE" "$CLONE_DIR/programs/winepath/winepath.exe.so" -w "$SCRIPT_DIR"`
    for opt in $MODULES; do
        if test -f "$SCRIPT_DIR/$opt.sh"; then
            sh "$SCRIPT_DIR/$opt.sh"
            TAG="$TAG-$opt"
        else
            echo "WARNING: Option module $opt not found"
        fi
    done
    # Always disable the crash dialog.
    "$WINE" "$CLONE_DIR/programs/regedit/regedit.exe.so" "$WINE_SCRIPT_DIR\\nocdlg.reg"
    #sleep 5
    if test `echo $TAG | wc -m` -gt 21; then
        TAG=`echo $TAG | cut -c 1-20`
    fi
    if test x$TIMEOUT = xindefinite; then
        "$WINE" "$WINETEST_BIN" -c -t $TAG -m $MAIL
    else
        # Set up a trap just in case...
        # Thanks to Jeremy White for the idea.
        # FIXME: Trap to what?
        set -m
        #trap cleanup CHLD
        #trap cleanup INT
        "$WINE" "$WINETEST_BIN" -c -t $TAG -m $MAIL &
        sleep $TIMEOUT
    fi
}

parse_args $@
if test ! -e $BUILD_DIR; then
    # Build directory doesn't exist; invoke build-wine-test to build it.
    export WINE_DIR BUILD_DIR CLONE_DIR
    if ! sh $WINE_DIR/build-wine-test.sh; then
        exit 1
    fi
fi
run_test

