/*
 * Wine bootup handler application
 * 
 * Copyright 2001 Andreas Mohr
 * To be distributed under the Wine License
 *
 * Used for running several Windows startup processes, e.g.:
 * - wininit.ini processing
 * - registry RenameFiles entries
 * - RunServices* / RunOnce* / Run registry keys
 * - win.ini Load= / Run= entries
 * - StartUp folder processing
 * - probably other stuff that's NIY
 *
 * For a description of some parts of the Windows bootup process,
 * see the following MS KB articles:
 * 
 * Q137367 Definition of the RunOnce Keys in the Registry
 * Q179365 Run, RunOnce, RunServices, RunServicesOnce and Startup
 * Q232487 Description of RunOnceEx Registry Key
 * Q232509 Syntax for the RunOnceEx Registry Key
 *
 * And for StartUp folder stuff, see:
 * http://cpcug.org/user/clemenzi/technical/Win_95_Registry.htm
 *
 * The autorun registry keys are all located in the following subkey string:
 * Software\Microsoft\Windows\CurrentVersion
 * ("KEY_CV" from now on)
 *
 * The bootup order is as follows:
 * 
 * - < DOS bootup >
 * 
 * - wininit.ini (done by DOS program wininit.exe)
 * 
 * - < GUI init >
 * 
 * - HKLM RunServicesOnce
 *   HKLM\KEY_CV\RunServicesOnce
 *   CWD: C:\ (hmm, or rather windows startup dir ? FIXME)
 *   
 * - HKLM RunServices
 *   HKLM\KEY_CV\RunServices
 *   CWD: dir where windows got started from (usually C:\)
 *   
 * *** stuff done by win32 program runonce.exe ***
 *
 * - RenameFiles
 *                
 * - HKLM RunOnce (not in NT 3.51 ! Waiting for program exit)
 *   HKLM\KEY_CV\RunOnce
 *   CWD: C:\ (hmm, or rather windows startup dir ? FIXME)
 * 
 * *** end runonce.exe ***
 * 
 * - HKLM RunOnceEx (from Win98 / "Windows Desktop Update" on only. Waiting
 * for program exit)
 *   HKLM\KEY_CV\RunOnceEx
 *   CWD: C:\ (hmm, or rather windows startup dir ? FIXME)
 *   
 * - < Logon Prompt >
 * 
 * - win.ini [Windows] Load= line (minimized loading !)
 *   CWD: program dir
 * 
 * - win.ini [Windows] Run= line (normal loading !)
 *   CWD: program dir
 * 
 * - HKLM Run
 *   HKLM\KEY_CV\Run
 *   CWD: C:\WINDOWS
 * 
 * - HKCU Run
 *   HKCU\KEY_CV\Run
 *   CWD: C:\WINDOWS
 * 
 * - Common Startup folder
 *   HKLM\KEY_CV\Explorer\Shell Folders\Common Startup
 *   CWD: program dir
 * 
 * - User Startup folder
 *   HKCU\KEY_CV\Explorer\Shell Folders\Startup
 *   CWD: FIXME !
 * 
 * - HKCU RunOnce
 *   HKCU\KEY_CV\RunOnce
 *   CWD: C:\WINDOWS
 *
 * known runonce.exe switches:
 * -a
 *   unknown behaviour, FIXME.
 * -m
 *   used by Windows on bootup:
 *   C:\WINDOWS\SYSTEM\runonce.exe -m
 *   CWD is C:\ (hmm, or rather windows startup dir ? FIXME)
 *   maybe -m means msgsvr32 ?? (when calling runonce.exe without -m,
 *   it complains that it's not being called via msgsvr32)
 * -q
 *   unknown behaviour, FIXME.
 *
 * FIXME:
 * - what about NT key "HKLM\CurrentControlSet\Control\Session Manager" ?
 * - PendingRenameOperations:REG_MULTI_SZ key ?
 *   Where exactly should this one be processed ?
 * - Where is DeleteFiles and PreConvRenameFiles being started ?
 * - What about HKCU RunServicesOnce ?
 * - RunOnce and RunOnceEx entries seem to be able to contain non-program
 *   entries, too, e.g.: "FlushRegistry"
 *
 * TODO (in order of importance):
 * - wininit.ini case insensitive !!
 * - every key getting deleted properly ?
 * - verify everything
 * - test verbose, quiet mode
 * - add some config variables to be able to specify external shell scripts
 *   for GUI mode interaction ? should be a cool idea.
 * - use proper CWDs when executing stuff
 * - execute native runonce.exe if available ? (at NO_RUNONCE_BINARY define)
 * - test misc. switches of runonce.exe
 * - check #includes needed
 * - grep for FIXME !!
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <windows.h>
#include "wineboot.h"
#include "regstr.h"
#include "file.h"

/* Work around a Wine bug which defines handles as UINT rather than LPVOID */
#ifdef WINE_STRICT
#define NULL_HANDLE NULL
#else
#define NULL_HANDLE 0
#endif

/* this needs to remain in this file due to unsupported yet variable argument
 * macros in wrc. ugh. */
#define MESSAGE(x...) fprintf(stderr, x)
#define TRACE(x...) if (BF_Verbose) fprintf(stderr, x)
#define WARN(x...) fprintf(stderr, x)
#define FIXME(x...) fprintf(stderr, x)
#define ERR(x...) fprintf(stderr, x)
 
 /* Filename for Profile_WinInit */
#define WININITINI "wininit.ini"
#define WININITBAK "wininit.bak"

static HKEY hkeyCfg = 0;
static HKEY hkeyHKLM_CV = 0;
static HKEY hkeyHKCU_CV = 0;

static char *do_it;
#if BOOT_PROFILING
static DWORD BOOT_startticks;
#endif
static BOOL BOOT_started = FALSE;
static BOOL BOOT_viruswarn = FALSE;

#define CFGS_0		0x0000
#define	CFGS_LOADED	0x0001
#define CFGS_NOTAVAIL	0x0002
#define CFGS_MODIFIED	0x0004

typedef struct {
  LPSTR name;
  BOOL value;
  DWORD status;
} BOOT_CONFIG;

BOOT_CONFIG BOOT_Config[] =
{
	/* enable the whole thing */
	{"BootConfig_Enable",			 TRUE, CFGS_0 },

	/* do a dry run, i.e. show what to do, but don't act */
	{"BootConfig_DryRun",			FALSE, CFGS_0 },

	/* extra verbosity on console */
	{"BootConfig_TTY_Verbose",		FALSE, CFGS_0 },

	/* extra quiet on console */
	{"BootConfig_TTY_Quiet",		FALSE, CFGS_0 },

	/* process wininit.ini ? */
	{"WininitIni_Execute",			 TRUE, CFGS_0 },

	{"RunServicesOnce_Execute",		 TRUE, CFGS_0 },
	{"RunServicesHKLM_Execute",		 TRUE, CFGS_0 },
	{"RenameFiles_Execute",			 TRUE, CFGS_0 },
	{"RunOnceHKLM_Execute",			 TRUE, CFGS_0 },
	{"RunOnceEx_Execute",			 TRUE, CFGS_0 },
	{"WinIniLoad_Execute",			 TRUE, CFGS_0 },
	{"WinIniRun_Execute",			 TRUE, CFGS_0 },
	{"RunHKLM_Execute",			 TRUE, CFGS_0 },
	{"RunHKCU_Execute",			 TRUE, CFGS_0 },
	{"StartupFolderCommon_Execute",		FALSE, CFGS_0 },
	{"StartupFolderUser_Execute",		FALSE, CFGS_0 },
	{"RunOnceHKCU_Execute",			 TRUE, CFGS_0 },
};

enum {
	CFG_ENABLE = 0,
	CFG_DRYRUN,
	CFG_TTY_VERBOSE,
	CFG_TTY_QUIET,
	CFG_WININIT_EXECUTE,
	CFG_RSO_EXECUTE,
	CFG_RSHKLM_EXECUTE,
	CFG_RENAMEFILES_EXECUTE,
	CFG_ROHKLM_EXECUTE,
	CFG_ROEX_EXECUTE,
	CFG_WININILOAD_EXECUTE,
	CFG_WININIRUN_EXECUTE,
	CFG_RUNHKLM_EXECUTE,
	CFG_RUNHKCU_EXECUTE,
	CFG_SFC_EXECUTE,
	CFG_SFU_EXECUTE,
	CFG_ROHKCU_EXECUTE,
};

static BOOL BF_DryRun;
static BOOL BF_Verbose;
static BOOL BF_Quiet;
	
typedef struct {
    LPSTR text_msg;
    LPSTR text_shutup;
    DWORD flags;
    DWORD retval;
} NB_DATA;

DWORD BOOT_NotifyBox_Show(LPSTR, LPSTR, DWORD);

/**************************** generic stuff start *****************************/
static BOOL BOOT_ReadConfig(DWORD what)
{
    char buffer[16];
    DWORD type, count = sizeof(buffer);
    BOOL active = FALSE;

    if (!hkeyCfg)
        if(RegOpenKeyA(HKEY_LOCAL_MACHINE, "Software\\Wine\\Wine\\Config\\wineboot", &hkeyCfg) != ERROR_SUCCESS)
    {
	TRACE("Failed to open wineboot config registry key !!\n");
	return 0;
    }
    buffer[0] = '\0';
    if (RegQueryValueExA(hkeyCfg, BOOT_Config[what].name, 0, &type, buffer, &count) == ERROR_SUCCESS)
    {
	switch (toupper(buffer[0]))
	{
	    case 'N':
	    case 'F':
	    case '0':
		active = FALSE;
		break;
	    case 'Y':
	    case 'T':
	    case '1':
		active = TRUE;
		break;
	}
	BOOT_Config[what].value = active;
	BOOT_Config[what].status |= CFGS_LOADED;
    }
    else
	BOOT_Config[what].status |= CFGS_NOTAVAIL;
    TRACE("Got config param '%s': '%s'\n", BOOT_Config[what].name, buffer);
    return BOOT_Config[what].value;
}

#ifdef WB_NOTYETUSED
static void BOOT_WriteConfig(DWORD what, BOOL value)
{
    char buffer[2];

    buffer[0] = (value) ? 'Y' : 'N';
    buffer[1] = '\0';

    if (RegSetValueExA(hkeyCfg, BOOT_Config[what].name, 0, REG_SZ, buffer, 2) != ERROR_SUCCESS)
	ERR("writing Config item '%s' failed !!\n", BOOT_Config[what].name);
}
#endif

static void BOOT_StartMessages(BOOL do_virus)
{
    if (!BOOT_started)
    {
#if BOOT_PROFILING
        BOOT_startticks = GetTickCount();
#endif
        if (!(BF_Quiet))
            MESSAGE("************************ Wine bootup processing starting ***********************\n");
        BOOT_started = TRUE;
    }

    /* we don't want Wine to become the primary choice of myriads of virii
     * after all the Outlock and IIS crap that happened in the last decade...
     */
    if ((!BOOT_viruswarn) && (do_virus))
    {
	/* yeah, yell at people in order to make a slight difference between
	 * possibly thousands of bootup messages... */
        MESSAGE("BOOT: WARNING: at least one permanent Run/RunServices/Run=/Load=/StartUp Autorun entry detected !\n*** MAKE SURE THAT THE PROGRAMS STARTED HERE ARE NO VIRII ! ***\nIn case you're suspicious, search e.g. www.google.com for the executable name.");
        BOOT_viruswarn = TRUE;
    }
}

static void BOOT_FinishMessage(void)
{
    if (!(BF_Quiet))
#if BOOT_PROFILING
        MESSAGE("************ Wine bootup processing finished (%08ld milliseconds) ***********\n", GetTickCount() - BOOT_startticks);
#else
        MESSAGE("*********************** Wine bootup processing finished ************************\n");
#endif
}

/* BOOT_RegStuff_AddToKill()
 * When enumerating keys, we may not disrupt our current enumeration's key
 * infrastructure. Thus we must delay deletion */
static void *del_buffer = NULL;
static DWORD del_buffer_size = 0;
static DWORD del_buffer_filled = 0;
void BOOT_RegStuff_AddToKill(HKEY hkeyParent, BOOL is_key, LPSTR what)
{
#define DEL_BUFFER_STEP	4096
    DWORD nextsize;
    char *p;

    TRACE("registering %s '%s' of hkey %p for delayed deletion.\n",
	is_key ? "subkey" : "value", what, hkeyParent);
    nextsize = del_buffer_filled + sizeof(hkeyParent) + sizeof(is_key) + strlen(what)+1;
    if (del_buffer_size < nextsize)
    {
	del_buffer = HeapReAlloc(GetProcessHeap(), 0, del_buffer, nextsize + DEL_BUFFER_STEP);
	if (!del_buffer)
	{
	    ERR("HeapReAlloc of del_buffer failed !!!\n");
	    del_buffer_size = del_buffer_filled = 0;
	    return;
	}
	else
            del_buffer_size = nextsize + DEL_BUFFER_STEP;
    }
    p = del_buffer;
    p += del_buffer_filled;

    *(HKEY *)p = hkeyParent;
    p += sizeof(HKEY);
    *(BOOL *)p = is_key;
    p += sizeof(BOOL);
    strcpy(p, what);
    p += strlen(what)+1;
    
    del_buffer_filled = nextsize;
}

void BOOT_RegStuff_KillDelayed()
{
    char *p;
    HKEY hkeyParent;
    BOOL is_key;
    
    if ((BF_DryRun) || (!del_buffer))
	return;
	    
    p = del_buffer;
    do {
	hkeyParent = *(HKEY *)p;
	p += sizeof(HKEY);
	is_key = *(BOOL *)p;
	p += sizeof(BOOL);
        TRACE("delayed deletion of %s '%s' of hkey %p.\n",
	    is_key ? "subkey" : "value", p, hkeyParent);
	if (is_key)
	    RegDeleteKeyA(hkeyParent, p);
	else
	    RegDeleteValueA(hkeyParent, p);
	p += strlen(p)+1;
    } while (p < (char *)(del_buffer + del_buffer_filled));

    del_buffer_filled = 0;
}

void BOOT_RegStuff_Cleanup()
{
    if (del_buffer)
	HeapFree(GetProcessHeap(), 0, del_buffer);
}

static BOOL BOOT_CreateProcess(LPSTR filename, LPSTR dir, BOOL wait, BOOL minimized)
{
    BOOL res;
    STARTUPINFOA si;
    PROCESS_INFORMATION info;
    DWORD exit_code;

    if (BF_DryRun)
	return TRUE;

    memset(&si, 0, sizeof(si));
    memset(&info, 0, sizeof(info));
    res = CreateProcessA(NULL, filename, NULL, NULL, FALSE, 0, NULL, dir, &si, &info);
    if ((wait) && (res == TRUE))
    {   /* wait for the process to exit */
        WaitForSingleObject(info.hProcess, INFINITE);
        res = GetExitCodeProcess(info.hProcess, &exit_code);
	return exit_code;
    }
    if ((!res) && (!(BF_Quiet)))
        MESSAGE("BOOT: failed to run '%s' !\n", filename);
        
    return res;
}

#define OPENREG_CURRVER(use_hkcu) \
    if ((use_hkcu && !hkeyHKCU_CV) || (!use_hkcu && !hkeyHKLM_CV)) \
	if (!BOOT_OpenReg_CurrVer(use_hkcu)) \
	    return; 
#define USE_HKCU	TRUE
#define USE_HKLM	FALSE

static BOOL BOOT_OpenReg_CurrVer(BOOL hkcu)
{
    HKEY hkeyroot = hkcu ? HKEY_CURRENT_USER : HKEY_LOCAL_MACHINE;
    HKEY *phkey = hkcu ? &hkeyHKCU_CV : &hkeyHKLM_CV;

    return (RegOpenKeyA(hkeyroot, "Software\\Microsoft\\Windows\\CurrentVersion", phkey) == ERROR_SUCCESS);
}

static void BOOT_CloseRegistry(void)
{
    int i;

    for (i=0; i < (sizeof(BOOT_Config)/sizeof(BOOT_CONFIG)); i++)
	WRITE_CONFIG(i);

    if (hkeyHKLM_CV) RegCloseKey(hkeyHKLM_CV);
    if (hkeyHKCU_CV) RegCloseKey(hkeyHKCU_CV);
    if (hkeyCfg) RegCloseKey(hkeyCfg);
}
/**************************** generic stuff end *******************************/


/**************************************************
 *         BOOT_ProcessWininitIni
 *
 * Process wininit.ini hopefully like wininit.exe does
 * Originally created by Uwe Bonnes
 */
static void BOOT_WininitIni(void)
{
    int i,j=0,k=0;
    char *buffer=NULL,*p;
    char inifile[MAX_PATHNAME_LEN],bakfile[MAX_PATHNAME_LEN];

    TRACE("%sProcessing wininit.ini\n", do_it);
    buffer = HeapReAlloc( GetProcessHeap(), 0, buffer,(j++ +1)*1024 );
    while (1)
    {
        if ((k) && !(BF_Quiet))
	    FIXME("Some previous programs produced wininit.ini in different case\n");
        i = GetPrivateProfileStringA(NULL,NULL,"",buffer,(j)*1024,WININITINI);
        if(i<2)
	{
	    TRACE("No (more) %s present\n",WININITINI);
	    break;
	}
        BOOT_StartMessages(VIRUS_NOWARN);

        /* Check if there are other sections in the ini file beside the rename section */
        for (p=buffer; *p; p= p + lstrlenA(p) +1)
	{
	    if (strcasecmp(p,"rename"))
	    {
	        WARN("Unknown section [%s]\n",p);
	        WARN("Processing anyway\n");
	    }
	}
        /* Maybe the section is damn long. Try to cope with that */
        while (1)
	{
	    i = GetPrivateProfileStringA("rename",NULL,"",buffer,(j)*1024,WININITINI);
	    if (i < ((j)*1024)-1)
	        break;
	    buffer = HeapReAlloc( GetProcessHeap(), 0, buffer,(j++ +1)*1024 );
	    if (!buffer)
	    {
	        ERR("Failed to allocate needed space for scanning %s\n",WININITINI);
	        return;
	    }
	}
        /* NUL means delete, otherwise move */
        for (p=buffer; *p; p= p + lstrlenA(p) +1)
	{
	    i = GetPrivateProfileStringA("rename",p,"",inifile,MAX_PATHNAME_LEN,WININITINI);
	    if (i > MAX_PATHNAME_LEN - 1)
	        WARN("Filename too long\n");
	    else
	    {
	        BOOL delete = (!strcasecmp(p,"NUL"));

	        if (delete)
	        {
		    TRACE("%sdeleting %s\n",do_it,inifile);
		    if (!(BF_DryRun))
		        DeleteFileA(inifile);
	        }
	        else
	        {
		    TRACE("%srenaming  %s to  %s\n",do_it,inifile,p);
		    if (!(BF_DryRun))
		        MoveFileExA(inifile,p,MOVEFILE_REPLACE_EXISTING);
	        }
	    }
	}
        GetWindowsDirectoryA(inifile,MAX_PATHNAME_LEN);
        lstrcpynA(inifile+lstrlenA(inifile),"\\",MAX_PATHNAME_LEN-lstrlenA(inifile));
        lstrcpynA(bakfile,inifile,MAX_PATHNAME_LEN);
        lstrcpynA(inifile+lstrlenA(inifile),WININITINI,MAX_PATHNAME_LEN-1-lstrlenA(inifile));
        lstrcpynA(bakfile+lstrlenA(bakfile),WININITBAK,MAX_PATHNAME_LEN-1-lstrlenA(bakfile));
        if (k)
	    sprintf(bakfile,"%s%d",bakfile,k);
        TRACE(" %sMoving %s to %s\n",do_it,inifile,bakfile);
	if (!(BF_DryRun))
            MoveFileExA(inifile,bakfile,MOVEFILE_REPLACE_EXISTING);
        k++;
    }
    TRACE("BOOT: %s processing finished.\n", WININITINI);
    return;
}

static void BOOT_Registry_FileOps(void)
{
    HKEY hkeyRF, hkey;
    DWORD type, count;
    CHAR subkey_buf[256], name_buf[256], value_buf[256];
    CHAR path[MAX_PATHNAME_LEN], src[MAX_PATHNAME_LEN], dest[MAX_PATHNAME_LEN];
    DWORD name_len, value_len, subkey_len;
    DWORD dwIndex, postpath = 0, flags, res;
    FILETIME ft;
    int i;
    LPSTR p;

    OPENREG_CURRVER(USE_HKLM);

    if (RegOpenKeyA(hkeyHKLM_CV, "RenameFiles", &hkeyRF) != ERROR_SUCCESS)
    {
	TRACE("RenameFiles key not found, nothing to do.\n");
	return;
    }

    BOOT_StartMessages(VIRUS_NOWARN);

    subkey_len = sizeof(subkey_buf);
    /* Traverse all RenameFiles subkey entries we have available */
    for( dwIndex=0;
         (res = RegEnumKeyExA( hkeyRF, dwIndex, subkey_buf, &subkey_len,
                        NULL, NULL, NULL, &ft )) != ERROR_NO_MORE_ITEMS;
         ++dwIndex, subkey_len = sizeof(subkey_buf) )
    {
	TRACE("BOOT: found RenameFiles subkey '%s', res %lx\n", subkey_buf, res);

        if (RegOpenKeyA(hkeyRF, subkey_buf, &hkey) != ERROR_SUCCESS)
	{
	    MESSAGE("BOOT: can't open RenameFiles '%s' subkey !!\n", subkey_buf);
	    continue;
	}

        count = sizeof(path);
        if ((RegQueryValueExA(hkey, NULL, 0, &type, path, &count) != ERROR_SUCCESS)
        || (type != REG_SZ))
	{
	    FIXME("RenameFiles subkey '%s': path setting not found. What to do ???\n", subkey_buf);
	    *path = '\0';
	}
	else
	{
	    MESSAGE("found path setting '%s'.\n", path);
	    postpath = strlen(path);
	    if (postpath)
	    {
		path[postpath++] = '\\';
		path[postpath] = '\0';
	    }
	    strcpy(src, path);
	    strcpy(dest, path);
	}
    
	i = 0;
        name_len = sizeof(name_buf);
        value_len = sizeof(value_buf);
    	while (RegEnumValueA(hkey, i++, name_buf, &name_len, NULL, &type, value_buf, &value_len) == ERROR_SUCCESS)
	{
	    if (*name_buf == '\0')
		goto next_entry;

	    if (!(BF_Quiet))
                MESSAGE("BOOT: %sprocessing RenameFiles subkey '%s' entry: '%s': '%s'\n",
		do_it, subkey_buf, name_buf, value_buf);

	    if ((p = strchr(value_buf, ',')))
	    {
		*p = '\0';
		p++;
		flags = atoi(p); /* file attribute flags: FILE_ATTRIBUTE_xxx */
		TRACE("found file attribute flags: 0x%lx !\n", flags);
	    }
	    else
		flags = 0;
	    strcpy(&src[postpath], name_buf);
	    if (value_buf[0] != '\0')
	        strcpy(&dest[postpath], value_buf);
	    else
	        strcpy(&dest[postpath], name_buf);

	    if (!(BF_DryRun))
	    {
		if (!MoveFileExA(src,dest,MOVEFILE_REPLACE_EXISTING))
	    	    if (!(BF_Quiet))
			ERR("Can't rename file '%s' as '%s' (GLE %ld) ! Please investigate.\n", src, dest, GetLastError());
		if (flags)
		    SetFileAttributesA(dest, flags);
	    }

next_entry:
            /* initialize lengths for new iteration */
            name_len = sizeof(name_buf);
            value_len = sizeof(value_buf);
	}
        RegCloseKey(hkey);

	BOOT_RegStuff_AddToKill(hkeyRF, TRUE, subkey_buf);
    }
    BOOT_RegStuff_KillDelayed();

    RegCloseKey(hkeyRF);
    RegDeleteKeyA(hkeyHKLM_CV, "RenameFiles");
}

static void BOOT_Registry_RunStuff(BOOL hkcu, BOOL runonce, BOOL services, BOOL wait_exit)
{
    HKEY hkey, hkeyCV;
    CHAR name_buf[256], value_buf[256], *run_what, *what_root, cwd[MAX_PATHNAME_LEN];
    DWORD name_len, value_len;
    DWORD type;
    int i;

    OPENREG_CURRVER(hkcu);

    if (services)
    {
	if (runonce)
	    run_what = "RunServicesOnce";
	else
	    run_what = "RunServices";
    }
    else
    {
	if (runonce)
	    run_what = "RunOnce";
	else
	    run_what = "Run";
    }
    
    hkeyCV = hkcu ? hkeyHKCU_CV : hkeyHKLM_CV;
    what_root = hkcu ? "HKCU" : "HKLM";
    if ((!services) && (!runonce)) /* Run */
        GetWindowsDirectoryA(cwd, sizeof(cwd));
    else
	strcpy(cwd, "C:\\");
    
    if (RegOpenKeyA(hkeyCV, run_what, &hkey))
    {
	TRACE("'%s' key not found, nothing to do.\n", run_what);
        return;
    }

    i = 0;
    name_len = sizeof(name_buf);
    value_len = sizeof(value_buf);
    while (RegEnumValueA(hkey, i++, name_buf, &name_len, NULL, &type, value_buf, &value_len) == ERROR_SUCCESS)
    {
	if (type == REG_SZ)
        {
            value_buf[sizeof(value_buf) - 1] = '\0';

            BOOT_StartMessages(VIRUS_WARN);

	    if (!(BF_Quiet))
                MESSAGE("BOOT: %srunning '%s %s' entry '%s': '%s'\n",
			do_it, what_root, run_what, name_buf, value_buf);
            BOOT_CreateProcess(value_buf, cwd, wait_exit, PROCESS_MAXIMIZED);
	}
	if (runonce)
	    /* tests indicate that deletion probably is unconditionally */
	    BOOT_RegStuff_AddToKill(hkey, FALSE, name_buf);

        /* initialize lengths for new iteration */
        name_len = sizeof(name_buf);
        value_len = sizeof(value_buf);
    }

    /* now that we're finished, kill accumulated entries */
    BOOT_RegStuff_KillDelayed();

    RegCloseKey(hkey);
}

static void BOOT_Registry_RunOnceEx(void)
{
    HKEY hkeyROEx, hkey, hkeyDepend = 0;
    DWORD type, count;
    CHAR name_buf[256], value_buf[256], subkey_buf[256];
    DWORD ROEx_flags = 0;
    CHAR ROEx_title[256]; /* FIXME ? */
    DWORD name_len, value_len, subkey_len;
    DWORD dwIndex;
    FILETIME ft;
    int i;

    TRACE("BOOT: FIXME: a lot of non-basic RunOnceEx functionality NIY !\n");
    
    OPENREG_CURRVER(USE_HKLM);

    if (RegOpenKeyA(hkeyHKLM_CV, "RunOnceEx", &hkeyROEx))
    {
	TRACE("BOOT: no RunOnceEx registry entries to process.\n");
        return;
    }

    BOOT_StartMessages(VIRUS_NOWARN);

    type = REG_DWORD;
    count = sizeof(ROEx_flags);
    if (RegQueryValueExA(hkeyROEx, "Flags", 0, &type, (LPBYTE)&ROEx_flags, &count) == ERROR_SUCCESS)
	TRACE("BOOT: got RunOnceEx \"Flags\" 0x%lx\n", ROEx_flags);

    type = REG_SZ;
    count = sizeof(ROEx_title);
    if (RegQueryValueExA(hkeyROEx, "Flags", 0, &type, ROEx_title, &count) == ERROR_SUCCESS)
	TRACE("BOOT: got RunOnceEx \"Title\" '%s'\n", ROEx_title);
    
    if (RegOpenKeyA(hkeyROEx, "Depend", &hkeyDepend))
	TRACE("BOOT: found RunOnceEx \"Depend\" section.\n");

    subkey_len = sizeof(subkey_buf);
	
    /* Traverse all ROEx subkey entries that we have available */
    for( dwIndex=0;
         RegEnumKeyExA( hkeyROEx, dwIndex, subkey_buf, &subkey_len,
                        NULL, NULL, NULL, &ft ) != ERROR_NO_MORE_ITEMS;
         ++dwIndex, subkey_len = sizeof(subkey_buf) )
    {
	/* skip "Depend" key */        
	if (!(strcasecmp(name_buf, "Depend")))
	    continue;

	TRACE("BOOT: found '%s'\n", subkey_buf);

        if (RegOpenKeyA(hkeyROEx, subkey_buf, &hkey) != ERROR_SUCCESS)
	{
	    MESSAGE("BOOT: can't open RunOnceEx '%s' subkey !!\n", subkey_buf);
	    continue;
	}

	i = 0;
        name_len = sizeof(name_buf);
        value_len = sizeof(value_buf);
    	while (RegEnumValueA(hkey, i++, name_buf, &name_len, NULL, &type, value_buf, &value_len) == ERROR_SUCCESS)
	{
	    /* we assume entries without '|' are normal program entries */
	    if (!(strchr(value_buf, '|')))
	    {
	        if (!(BF_Quiet))
                    MESSAGE("BOOT: %srunning RunOnceEx entry '%s': '%s'\n",
		do_it, name_buf, value_buf);
                BOOT_CreateProcess(value_buf, "C:\\", PROCESS_WAITEXIT, PROCESS_MAXIMIZED);
	    }
	    else
		MESSAGE("BOOT: FIXME: HKLM RunOnceEx value '%s' ('%s') execution NIY !\n", name_buf, value_buf);

            /* initialize lengths for new iteration */
            name_len = sizeof(name_buf);
            value_len = sizeof(value_buf);
	}
	RegCloseKey(hkey);

	BOOT_RegStuff_AddToKill(hkeyROEx, TRUE, subkey_buf);
    }
    BOOT_RegStuff_KillDelayed();

    if (hkeyDepend)
	RegCloseKey(hkeyDepend);

    RegCloseKey(hkeyROEx);
}

static void BOOT_WinIni_Execute(BOOL do_load)
{
    char line_buf[1024]; /* FIXME ? */
    DWORD numchars;
    char buf[MAX_PATH]; /* FIXME ? */
    char *p1, *p2;

    numchars = GetProfileStringA("windows", do_load ? "Load" : "Run", "",
		    			line_buf, sizeof(line_buf));
    if (numchars)
    {
        BOOT_StartMessages(VIRUS_WARN);
	p1 = p2 = line_buf;

	/* split the program entries in this line and run them */
	while (1)
	{
	    while ((*p2 != ' ') && (*p2 != '\0') && (*p2 != '\t'))
		p2++;
	    if ((int)p2 - (int)p1 > sizeof(buf) - 1)
		FIXME("buffer problem !!\n");
	    strncpy(buf, p1, (int)p2 - (int)p1);
	    buf[(int)p2 - (int)p1] = '\0';
	    
	    if (!(BF_Quiet))
                MESSAGE("BOOT: %srunning win.ini %s= program entry '%s'\n",
			do_it, do_load ? "Load" : "Run", buf);
	
	    BOOT_CreateProcess(buf, NULL, PROCESS_NOWAITEXIT, do_load);

	    if (*p2 == '\0') /* abort if end of line */
		break;
	    p2++;
	    p1 = p2;
	}
    }
    return;
}

static void BOOT_StartUpFolder(BOOL common_startup)
{
    HKEY hkey = 0, hkeyCV;
    char *what_folder = (common_startup) ? "Common StartUp" : "StartUp";
    char startupdir[MAX_PATHNAME_LEN], buffer[MAX_PATHNAME_LEN]; /* FIXME: that ok ? */
    DWORD type, count;
    
    OPENREG_CURRVER( common_startup ? USE_HKLM : USE_HKCU);

    hkeyCV = common_startup ? hkeyHKLM_CV : hkeyHKCU_CV;
    
    if(RegOpenKeyA(hkeyCV, "Explorer\\Shell Folders", &hkey) != ERROR_SUCCESS)
	goto cleanup;

    count = sizeof(startupdir);
    if (RegQueryValueExA(hkey, what_folder, 0, &type, startupdir, &count) != ERROR_SUCCESS)
	goto cleanup;

    strcpy(buffer, startupdir);

    /* check whether \\*.* can be added to string */
    if (strlen(buffer)+1 < sizeof(buffer)-4)
    {
	HANDLE hFF;
	WIN32_FIND_DATAA fd;
	char *file_part, *p;

	/* add a marker for the file part */
	file_part = buffer+strlen(buffer);
	*file_part = '\\'; file_part++;

	*file_part = '\0'; strcat(file_part, "*.*");
	
	hFF = FindFirstFileA(buffer, &fd);
	if (hFF != INVALID_HANDLE_VALUE)
	do
	{
	    if (!(fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))
	    {
		p = strrchr(fd.cFileName, '.');

		if (p)
		{
		    p++;

		    if (!(strcasecmp(p, "exe")))
		    {
        		BOOT_StartMessages(VIRUS_WARN);
	    		if (!(BF_Quiet))
			    MESSAGE("BOOT: %srunning file '%s' in %s folder\n", do_it, fd.cFileName, what_folder);
			*file_part = '\0'; strcat(file_part, fd.cFileName);

			BOOT_CreateProcess(buffer, startupdir, PROCESS_NOWAITEXIT, PROCESS_MAXIMIZED);
		    }
		    else
	    		if (!(BF_Quiet))
           		    FIXME("Processing of %s folder file '%s' NIY !\n", what_folder, fd.cFileName);
		}
	    }
	}
	while (FindNextFileA(hFF, &fd));
    }
    else
	FIXME("buffer size problem !!\n");

cleanup:
    if (hkey) RegCloseKey(hkey);
    
    return;
}

int WINAPI WinMain( HINSTANCE hInst, HINSTANCE hPrevInst, LPSTR cmdline, int cmdshow )
{
/* FIXME */
#if 0
    /*------------------------------------------------------------------------
    ** Do the magic check
    **----------------------------------------------------------------------*/
    if ((!cmdline) || (strlen(cmdline) < 9) || (memcmp(cmdline, "--yesdoit", 9)))
    {
	MESSAGE("Sorry, no go.\nThis program is intended to be called by Wine on bootup in order to do a lot of program startup tasks.\nIf you still intend to run this program standalone, then run it using 'wineboot -- --yesdoit'.\n");
        return 0;
    }
#endif

    if (!GET_CONFIG(CFG_ENABLE))
    {
	MESSAGE("Boot handling disabled in wine registry, won't proceed.\n");
	return 0;
    }

    BF_DryRun = GET_CONFIG(CFG_DRYRUN);
    BF_Verbose = GET_CONFIG(CFG_TTY_VERBOSE);
    BF_Quiet = GET_CONFIG(CFG_TTY_QUIET);

    do_it = (BF_DryRun) ? "Not " : "";

    if (GET_CONFIG(CFG_TTY_VERBOSE))
        BOOT_StartMessages(VIRUS_NOWARN);


    if (GET_CONFIG(CFG_WININIT_EXECUTE))
        BOOT_WininitIni();

    if (GET_CONFIG(CFG_RSO_EXECUTE))
        BOOT_Registry_RunStuff(USE_HKLM, REG_RUNONCE, TRUE, PROCESS_NOWAITEXIT);
    if (GET_CONFIG(CFG_RSHKLM_EXECUTE))
        BOOT_Registry_RunStuff(USE_HKLM, REG_RUN, TRUE, PROCESS_NOWAITEXIT);

#if NO_RUNONCE_BINARY
    /* builtin runonce.exe begin */
    if (GET_CONFIG(CFG_RENAMEFILES_EXECUTE))
        BOOT_Registry_FileOps();

    if (GET_CONFIG(CFG_ROHKLM_EXECUTE))
        BOOT_Registry_RunStuff(USE_HKLM, REG_RUNONCE, FALSE, PROCESS_WAITEXIT);
    /* runonce.exe end */
#else
    /* execute "runonce.exe -m", native Windows binary */
#endif

    if (GET_CONFIG(CFG_ROEX_EXECUTE))
	BOOT_Registry_RunOnceEx();
    
    if (GET_CONFIG(CFG_WININILOAD_EXECUTE))
        BOOT_WinIni_Execute(WININI_LOAD);
    if (GET_CONFIG(CFG_WININIRUN_EXECUTE))
        BOOT_WinIni_Execute(WININI_RUN);

    if (GET_CONFIG(CFG_RUNHKLM_EXECUTE))
        BOOT_Registry_RunStuff(USE_HKLM, REG_RUN, FALSE, PROCESS_NOWAITEXIT);
    if (GET_CONFIG(CFG_RUNHKCU_EXECUTE))
        BOOT_Registry_RunStuff(USE_HKCU, REG_RUN, FALSE, PROCESS_NOWAITEXIT);
	
    if (GET_CONFIG(CFG_SFC_EXECUTE))
        BOOT_StartUpFolder(STARTUP_COMMON);
    if (GET_CONFIG(CFG_SFU_EXECUTE))
        BOOT_StartUpFolder(STARTUP_USER);

    if (GET_CONFIG(CFG_ROHKCU_EXECUTE))
        BOOT_Registry_RunStuff(USE_HKCU, REG_RUNONCE, FALSE, PROCESS_NOWAITEXIT);

    BOOT_RegStuff_Cleanup();

    BOOT_CloseRegistry();

    if (BOOT_started)
        BOOT_FinishMessage();
    
    return 1;
}
